"""
Excel Handler để đọc và ghi Excel files
"""

import logging
from io import BytesIO
from typing import Any

logger = logging.getLogger(__name__)

# Lazy loading - chỉ import pandas khi cần
_pandas = None


def _get_pandas():
    """Lazy load pandas"""
    global _pandas
    if _pandas is None:
        import pandas as pd
        _pandas = pd
    return _pandas


def read_excel(file_content: bytes, question_column: str = "Câu hỏi") -> list[dict[str, Any]]:
    """
    Đọc Excel file và extract câu hỏi

    Args:
        file_content: File content dạng bytes
        question_column: Tên cột chứa câu hỏi

    Returns:
        List dicts: [{"id": 1, "question": "...", "row_data": {...}}, ...]
    """
    try:
        pd = _get_pandas()
        df = pd.read_excel(BytesIO(file_content))

        # Tìm cột câu hỏi (case-insensitive)
        question_col = None
        for col in df.columns:
            if question_column.lower() in col.lower() or "question" in col.lower() or "query" in col.lower():
                question_col = col
                break

        if question_col is None:
            raise ValueError(f"Không tìm thấy cột '{question_column}' trong file Excel")

        results = []
        for idx, row in df.iterrows():
            question = str(row[question_col]).strip()
            if not question or question.lower() in ["nan", "none", ""]:
                continue

            # Lưu toàn bộ row data để giữ nguyên các cột khác
            row_data = row.to_dict()
            # Convert idx to int (pandas index can be various types)
            row_id = int(idx) + 1 if isinstance(idx, (int, float)) else len(results) + 1
            results.append(
                {
                    "id": row_id,  # 1-based index
                    "question": question,
                    "row_data": row_data,
                }
            )

        logger.info(f"Đọc được {len(results)} câu hỏi từ Excel")
        return results

    except Exception as e:
        logger.error(f"Error reading Excel: {e}", exc_info=True)
        raise


def create_results_excel(
    summary_data: dict[str, Any],
    detailed_results: list[dict[str, Any]],
    aggregated_results: list[dict[str, Any]],
) -> bytes:
    """
    Tạo Excel file với 3 sheets: Summary, Results, Aggregated

    Args:
        summary_data: Dict tổng kết
        detailed_results: List kết quả chi tiết từng lần test
        aggregated_results: List kết quả tổng hợp theo câu hỏi

    Returns:
        Excel file dạng bytes
    """
    try:
        pd = _get_pandas()
        output = BytesIO()

        # Type ignore for BytesIO - pandas accepts it at runtime
        with pd.ExcelWriter(output, engine="openpyxl") as writer:  # type: ignore
            # Sheet 1: Summary
            summary_df = pd.DataFrame([summary_data])
            summary_df.to_excel(writer, sheet_name="Summary", index=False)

            # Sheet 2: Results (chi tiết từng lần test)
            if detailed_results:
                results_df = pd.DataFrame(detailed_results)
                results_df.to_excel(writer, sheet_name="Results", index=False)

            # Sheet 3: Aggregated (tổng hợp theo câu hỏi)
            if aggregated_results:
                aggregated_df = pd.DataFrame(aggregated_results)
                aggregated_df.to_excel(writer, sheet_name="Aggregated", index=False)

        output.seek(0)
        return output.getvalue()

    except Exception as e:
        logger.error(f"Error creating Excel: {e}", exc_info=True)
        raise

