"""
Excel Handler để đọc và ghi Excel files
"""

import logging
from io import BytesIO
from typing import Any

logger = logging.getLogger(__name__)

# Lazy loading - chỉ import pandas khi cần
_pandas = None


def _get_pandas():
    """Lazy load pandas"""
    global _pandas
    if _pandas is None:
        import pandas as pd

        _pandas = pd
    return _pandas


def read_excel(
    file_content: bytes, question_column: str = "Câu hỏi"
) -> list[dict[str, Any]]:
    """
    Đọc Excel file và extract câu hỏi

    Args:
        file_content: File content dạng bytes
        question_column: Tên cột chứa câu hỏi

    Returns:
        List dicts: [{"id": 1, "question": "...", "row_data": {...}}, ...]
    """
    try:
        pd = _get_pandas()
        df = pd.read_excel(BytesIO(file_content))

        # Tìm cột câu hỏi (case-insensitive)
        question_col = None
        for col in df.columns:
            if (
                question_column.lower() in col.lower()
                or "question" in col.lower()
                or "query" in col.lower()
            ):
                question_col = col
                break

        # Nếu chỉ có 1 cột trong file, mặc định dùng cột đó làm question
        if question_col is None:
            if df.shape[1] == 1:
                question_col = df.columns[0]
            else:
                raise ValueError(
                    f"Không tìm thấy cột '{question_column}' trong file Excel"
                )

        results = []
        for idx, row in df.iterrows():
            question = str(row[question_col]).strip()
            if not question or question.lower() in ["nan", "none", ""]:
                continue

            # Lưu toàn bộ row data để giữ nguyên các cột khác
            row_data = row.to_dict()
            # Convert idx to int (pandas index can be various types)
            row_id = int(idx) + 1 if isinstance(idx, (int, float)) else len(results) + 1
            results.append(
                {
                    "id": row_id,  # 1-based index
                    "question": question,
                    "row_data": row_data,
                }
            )

        logger.info(f"Đọc được {len(results)} câu hỏi từ Excel")
        return results

    except Exception as e:
        logger.error(f"Error reading Excel: {e}", exc_info=True)
        raise


def create_results_excel(
    summary_data: dict[str, Any],
    detailed_results: list[dict[str, Any]],
    aggregated_results: list[dict[str, Any]],
) -> bytes:
    """
    Tạo Excel file với 1 sheet duy nhất:
    - Cột Question (câu hỏi)
    - Cột Answer1, Answer2, ... (theo số lần test)

    Args:
        summary_data: Dict tổng kết
        detailed_results: List kết quả chi tiết từng lần test
        aggregated_results: List kết quả tổng hợp theo câu hỏi

    Returns:
        Excel file dạng bytes
    """
    try:
        pd = _get_pandas()
        output = BytesIO()

        # Lấy số lần test từ summary
        num_tests = int(summary_data.get("num_tests_per_question", 1))

        # Xây dựng dữ liệu cho sheet: mỗi row là 1 câu hỏi + các answers
        sheet_data: list[dict[str, Any]] = []
        for agg in aggregated_results:
            row: dict[str, Any] = {"Question": agg.get("question", "")}
            question_id = agg.get("question_id")

            for test_num in range(1, num_tests + 1):
                # Tìm kết quả của lần test này trong detailed_results
                result = next(
                    (
                        r
                        for r in detailed_results
                        if r.get("question_id") == question_id
                        and r.get("test_attempt") == test_num
                    ),
                    None,
                )

                answer = result.get("ai_response", "") if result else ""
                row[f"Answer {test_num}"] = answer

            sheet_data.append(row)

        # Tạo DataFrame và ghi ra Excel
        results_df = pd.DataFrame(sheet_data)

        with pd.ExcelWriter(output, engine="openpyxl") as writer:  # type: ignore
            results_df.to_excel(writer, sheet_name="Results", index=False)

            # Format column width
            worksheet = writer.sheets["Results"]
            for column_cells in worksheet.columns:
                max_length = 0
                column_letter = column_cells[0].column_letter
                for cell in column_cells:
                    try:
                        if cell.value:
                            max_length = max(max_length, len(str(cell.value)))
                    except Exception:
                        pass
                adjusted_width = min(max_length + 2, 50)
                worksheet.column_dimensions[column_letter].width = adjusted_width

        output.seek(0)
        return output.getvalue()

    except Exception as e:
        logger.error(f"Error creating Excel: {e}", exc_info=True)
        raise
