"""
FastAPI routes cho batch testing tool
"""

import logging
import uuid
from typing import Any

from fastapi import APIRouter, BackgroundTasks, File, Form, HTTPException, UploadFile
from fastapi.responses import FileResponse, JSONResponse

from services.batch_processor import BatchProcessor
from utils.excel_handler import read_excel, create_results_excel

logger = logging.getLogger(__name__)
router = APIRouter()

# In-memory storage cho progress (trong production nên dùng Redis hoặc DB)
progress_store: dict[str, dict[str, Any]] = {}


@router.post("/api/batch-test/upload", summary="Upload Excel và bắt đầu batch test")
async def upload_and_test(
    file: UploadFile = File(...),
    num_tests: int = Form(1),
    question_column: str = Form("Câu hỏi"),
):
    """
    Upload Excel file và bắt đầu batch testing

    Args:
        file: Excel file
        num_tests: Số lần test mỗi câu hỏi
        question_column: Tên cột chứa câu hỏi

    Returns:
        Task ID để track progress
    """
    try:
        # Validate file
        if not file.filename or not file.filename.endswith((".xlsx", ".xls")):
            raise HTTPException(status_code=400, detail="File phải là Excel (.xlsx hoặc .xls)")

        # Đọc file
        file_content = await file.read()
        questions = read_excel(file_content, question_column=question_column)

        if not questions:
            raise HTTPException(status_code=400, detail="Không tìm thấy câu hỏi nào trong file")

        # Tạo task ID
        task_id = str(uuid.uuid4())

        # Initialize progress
        progress_store[task_id] = {
            "status": "processing",
            "processed": 0,
            "total": len(questions) * num_tests,
            "current_question": 0,
            "current_attempt": 0,
            "successful": 0,
            "failed": 0,
        }

        # Start batch processing (async)
        processor = BatchProcessor()

        async def process_task():
            try:
                def progress_callback(progress: dict[str, Any]):
                    progress_store[task_id].update(progress)

                result = await processor.process_batch(
                    questions=questions,
                    num_tests_per_question=num_tests,
                    progress_callback=progress_callback,
                )

                # Tạo Excel output
                excel_output = create_results_excel(
                    summary_data=result["summary"],
                    detailed_results=result["detailed_results"],
                    aggregated_results=result["aggregated_results"],
                )

                # Lưu kết quả
                progress_store[task_id].update(
                    {
                        "status": "completed",
                        "result": result,
                        "excel_output": excel_output,
                    }
                )

            except Exception as e:
                logger.error(f"Error processing batch: {e}", exc_info=True)
                progress_store[task_id].update(
                    {
                        "status": "error",
                        "error_message": str(e),
                    }
                )

        # Run async task
        import asyncio

        asyncio.create_task(process_task())

        return JSONResponse(
            {
                "task_id": task_id,
                "message": "Batch testing đã bắt đầu",
                "total_questions": len(questions),
                "num_tests_per_question": num_tests,
                "total_tests": len(questions) * num_tests,
            }
        )

    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"Error in upload_and_test: {e}", exc_info=True)
        raise HTTPException(status_code=500, detail=str(e))


@router.get("/api/batch-test/progress/{task_id}", summary="Lấy progress của batch test")
async def get_progress(task_id: str):
    """
    Lấy progress của batch test

    Args:
        task_id: Task ID từ upload endpoint

    Returns:
        Progress info
    """
    if task_id not in progress_store:
        raise HTTPException(status_code=404, detail="Task không tồn tại")

    progress = progress_store[task_id].copy()

    # Không trả về excel_output trong progress (quá lớn)
    if "excel_output" in progress:
        progress["excel_ready"] = True
        del progress["excel_output"]

    return JSONResponse(progress)


@router.get("/api/batch-test/download/{task_id}", summary="Download Excel kết quả")
async def download_results(task_id: str, background_tasks: BackgroundTasks):
    """
    Download Excel file kết quả

    Args:
        task_id: Task ID từ upload endpoint
        background_tasks: Background tasks để xóa file sau khi download

    Returns:
        Excel file
    """
    import os
    import tempfile

    if task_id not in progress_store:
        raise HTTPException(status_code=404, detail="Task không tồn tại")

    task_data = progress_store[task_id]

    if task_data["status"] != "completed":
        raise HTTPException(status_code=400, detail="Task chưa hoàn thành")

    if "excel_output" not in task_data:
        raise HTTPException(status_code=500, detail="Excel output không tồn tại")

    # Tạo temporary file
    with tempfile.NamedTemporaryFile(delete=False, suffix=".xlsx") as tmp_file:
        tmp_file.write(task_data["excel_output"])
        tmp_path = tmp_file.name

    # Xóa file sau khi download
    background_tasks.add_task(os.unlink, tmp_path)

    return FileResponse(
        tmp_path,
        media_type="application/vnd.openxmlformats-officedocument.spreadsheetml.sheet",
        filename=f"batch_test_results_{task_id}.xlsx",
    )


@router.get("/api/batch-test/health", summary="Health check")
async def health_check():
    """Health check endpoint"""
    return JSONResponse({"status": "ok", "message": "Batch testing tool is running"})

