"""
Langfuse Client để lấy metrics từ Langfuse API (optional)
"""

import logging
from typing import Any

import httpx

from config import LANGFUSE_BASE_URL, LANGFUSE_SECRET_KEY

logger = logging.getLogger(__name__)


class LangfuseClient:
    """Client để lấy metrics từ Langfuse"""

    def __init__(self):
        self.base_url = LANGFUSE_BASE_URL or "https://cloud.langfuse.com"
        self.secret_key = LANGFUSE_SECRET_KEY
        self.enabled = bool(self.secret_key)

    async def get_trace_metrics(self, trace_id: str) -> dict[str, Any] | None:
        """
        Lấy metrics từ Langfuse trace

        Args:
            trace_id: Langfuse trace ID

        Returns:
            Dict metrics hoặc None nếu không lấy được
        """
        if not self.enabled:
            return None

        try:
            url = f"{self.base_url}/api/public/traces/{trace_id}"
            headers = {"Authorization": f"Bearer {self.secret_key}"}

            async with httpx.AsyncClient(timeout=10) as client:
                response = await client.get(url, headers=headers)

            if response.status_code == 200:
                data = response.json()
                latency = data.get("latency")
                return {
                    "cost": data.get("totalCost"),
                    "latency_ms": float(latency * 1000) if latency else None,
                    "input_tokens": data.get("inputTokens"),
                    "output_tokens": data.get("outputTokens"),
                    "total_tokens": data.get("totalTokens"),
                    "model": data.get("model"),
                }
            else:
                logger.warning(f"Failed to get trace {trace_id}: {response.status_code}")
                return None

        except Exception as e:
            logger.warning(f"Error getting Langfuse metrics: {e}")
            return None

    async def search_traces(
        self,
        user_id: str | None = None,
        session_id: str | None = None,
        limit: int = 10,
    ) -> list[dict[str, Any]]:
        """
        Tìm traces từ Langfuse theo user_id hoặc session_id

        Args:
            user_id: User ID để filter
            session_id: Session ID để filter
            limit: Số lượng traces tối đa

        Returns:
            List traces
        """
        if not self.enabled:
            return []

        try:
            url = f"{self.base_url}/api/public/traces"
            headers = {"Authorization": f"Bearer {self.secret_key}"}
            params: dict[str, Any] = {"limit": limit}

            if user_id:
                params["userId"] = user_id
            if session_id:
                params["sessionId"] = session_id

            async with httpx.AsyncClient(timeout=10) as client:
                response = await client.get(url, headers=headers, params=params)

                if response.status_code == 200:
                    data = response.json()
                    return data.get("data", [])
                else:
                    logger.warning(f"Failed to search traces: {response.status_code}")
                    return []

        except Exception as e:
            logger.warning(f"Error searching Langfuse traces: {e}")
            return []

