"""
API Client để gọi chatbot-canifa API
"""

import asyncio
import logging
import time
from typing import Any

import httpx
from config import CHATBOT_API_ENDPOINT, CHATBOT_API_URL, REQUEST_TIMEOUT

logger = logging.getLogger(__name__)


class ChatbotAPIClient:
    """Client để gọi chatbot API"""

    def __init__(self, api_url: str | None = None):
        self.base_url = api_url or CHATBOT_API_URL
        self.endpoint = CHATBOT_API_ENDPOINT
        self.timeout = REQUEST_TIMEOUT

    async def chat(
        self,
        query: str,
        user_id: str | None = None,
        test_id: str | None = None,
        test_attempt: int | None = None,
    ) -> dict[str, Any]:
        """
        Gọi chatbot API và trả về response với metrics

        Args:
            query: Câu hỏi
            user_id: User ID (optional)
            test_id: Test ID để track (optional)
            test_attempt: Số lần test (1, 2, 3...) (optional)

        Returns:
            Dict chứa response và metrics:
            {
                "status": "success" | "error",
                "ai_response": str,
                "product_ids": list,
                "latency_ms": float,
                "timestamp": str,
                "error_message": str | None
            }
        """
        if not user_id:
            user_id = (
                f"batch_test_{test_id}_{test_attempt}" if test_id else "batch_test_user"
            )

        # Nếu api_url đã chứa endpoint (kết thúc bằng /chat), dùng trực tiếp
        # Nếu không, thêm endpoint vào
        if self.base_url.endswith("/chat") or self.base_url.endswith("/api/agent/chat"):
            url = self.base_url
        else:
            url = f"{self.base_url}{self.endpoint}"

        payload = {
            "user_query": query,
            "user_id": user_id,
        }

        start_time = time.time()
        error_message = None
        status = "error"
        ai_response = ""
        product_ids = []

        try:
            async with httpx.AsyncClient(timeout=self.timeout) as client:
                response = await client.post(url, json=payload)
                latency_ms = (time.time() - start_time) * 1000

                if response.status_code == 200:
                    data = response.json()
                    status = data.get("status", "error")
                    ai_response = data.get("ai_response", "")
                    product_ids = data.get("product_ids", [])
                else:
                    error_message = (
                        f"API returned status {response.status_code}: {response.text}"
                    )
                    logger.error(error_message)

        except httpx.TimeoutException:
            latency_ms = (time.time() - start_time) * 1000
            error_message = f"Request timeout after {self.timeout}s"
            logger.error(error_message)
        except Exception as e:
            latency_ms = (time.time() - start_time) * 1000
            error_message = str(e)
            logger.error(f"Error calling chatbot API: {e}", exc_info=True)

        return {
            "status": status,
            "ai_response": ai_response,
            "product_ids": product_ids,
            "latency_ms": round(latency_ms, 2),
            "timestamp": time.strftime("%Y-%m-%d %H:%M:%S"),
            "error_message": error_message,
        }

    async def batch_chat(
        self,
        queries: list[str],
        user_id_prefix: str = "batch_test",
        max_concurrent: int = 5,
    ) -> list[dict[str, Any]]:
        """
        Gọi API cho nhiều queries song song

        Args:
            queries: List câu hỏi
            user_id_prefix: Prefix cho user_id
            max_concurrent: Số requests đồng thời tối đa

        Returns:
            List kết quả theo thứ tự queries
        """
        semaphore = asyncio.Semaphore(max_concurrent)

        async def chat_with_semaphore(index: int, query: str):
            async with semaphore:
                return await self.chat(query, user_id=f"{user_id_prefix}_{index}")

        tasks = [chat_with_semaphore(i, query) for i, query in enumerate(queries)]
        results = await asyncio.gather(*tasks)

        return results
