import json
import logging
from typing import Any

import httpx
from langchain_core.tools import tool
from pydantic import BaseModel, Field

from agent.prompt_utils import read_tool_prompt
from config import INTERNAL_STOCK_API

logger = logging.getLogger(__name__)

DEFAULT_MAX_SKUS = 200
DEFAULT_CHUNK_SIZE = 50


class StockCheckInput(BaseModel):
    skus: str = Field(
        description=(
            "Danh sách mã sản phẩm cần kiểm tra tồn kho (có thể là mã base, mã màu, "
            "hoặc SKU đầy đủ), phân cách bằng dấu phẩy. "
            "Ví dụ: '6ST25W005,6ST25W005-SE091,6ST25W005-SE091-L'"
        )
    )
    sizes: str | None = Field(
        default=None,
        description="Optional: lọc theo size (S,M,L,XL,140...)",
    )
    max_skus: int = Field(default=DEFAULT_MAX_SKUS, ge=1)
    chunk_size: int = Field(default=DEFAULT_CHUNK_SIZE, ge=1)
    timeout_sec: float = Field(default=10.0, gt=0)


def _split_csv(value: str | None) -> list[str]:
    if not value:
        return []
    return [token.strip() for token in value.split(",") if token.strip()]


def _normalize_size(token: str) -> str:
    normalized = token.strip().upper()
    if normalized.endswith("CM"):
        normalized = normalized[:-2]
    return normalized


def _is_full_sku(token: str) -> bool:
    return token.count("-") >= 2


async def _fetch_variants(codes: list[str]) -> list[dict[str, Any]]:
    if not codes:
        return []

    placeholders = ",".join(["%s"] * len(codes))
    sql = f"""
    SELECT
        internal_ref_code,
        magento_ref_code,
        product_color_code,
        size_scale
    FROM {TABLE_NAME}
    WHERE internal_ref_code IN ({placeholders})
       OR magento_ref_code IN ({placeholders})
       OR product_color_code IN ({placeholders})
    GROUP BY internal_ref_code, magento_ref_code, product_color_code, size_scale
    """

    params = codes * 3
    db = StarRocksConnection()
    return await db.execute_query_async(sql, params=tuple(params))


@tool("check_is_stock", args_schema=StockCheckInput)
async def check_is_stock(
    skus: str,
    sizes: str | None = None,
    max_skus: int = DEFAULT_MAX_SKUS,
    chunk_size: int = DEFAULT_CHUNK_SIZE,
    timeout_sec: float = 10.0,
) -> str:
    """
    Kiểm tra tồn kho theo mã sản phẩm.
    - Hỗ trợ mã base / mã màu / SKU đầy đủ.
    - Nếu thiếu màu/size thì tự expand từ DB, kết hợp màu + size (kể cả size số).
    - Gọi API tồn kho theo batch và trả về JSON tổng hợp.
    """
    if not skus:
        return "Lỗi: thiếu mã sản phẩm để kiểm tra tồn kho."

    api_url = f"{INTERNAL_STOCK_API}"
    
    payload = {
        "codes": skus,
        "sizes": sizes,
        "max_skus": max_skus,
        "chunk_size": chunk_size,
        "truncate": True,
        "expand_only": False 
    }

    try:
        async with httpx.AsyncClient(timeout=timeout_sec) as client:
            resp = await client.post(api_url, json=payload)
            resp.raise_for_status()
            return json.dumps(resp.json(), ensure_ascii=False)
            
    except httpx.RequestError as exc:
        logger.error(f"Network error checking stock: {exc}")
        return f"Lỗi kết nối khi kiểm tra tồn kho: {exc}"
    except httpx.HTTPStatusError as exc:
        logger.error(f"HTTP error {exc.response.status_code}: {exc}")
        return f"Lỗi server khi kiểm tra tồn kho (Status {exc.response.status_code})"
    except Exception as exc:
        logger.error(f"Unexpected error in check_is_stock: {exc}")
        return f"Lỗi không xác định khi kiểm tra tồn kho: {exc}"


# Load dynamic docstring from file
dynamic_prompt = read_tool_prompt("check_is_stock")
if dynamic_prompt:
    check_is_stock.__doc__ = dynamic_prompt
    check_is_stock.description = dynamic_prompt
