from fastapi import APIRouter, HTTPException, BackgroundTasks
from pydantic import BaseModel
from typing import List, Optional
import time
import random

router = APIRouter()

# --- MODELS ---
class MockQueryRequest(BaseModel):
    user_query: str
    user_id: Optional[str] = "test_user"
    session_id: Optional[str] = None

class MockDBRequest(BaseModel):
    vector: Optional[List[float]] = None
    top_k: int = 10

# --- MOCK DATA ---
MOCK_PRODUCTS = [
    {
        "internal_ref_code": "8TE24W001",
        "description_text_full": "Áo phông nam cotton thoáng mát, form regular fit.",
        "sale_price": 199000.0,
        "original_price": 299000.0,
        "discount_amount": 100000.0,
        "max_score": 0.89
    },
    {
        "internal_ref_code": "8JE24W002", 
        "description_text_full": "Quần Jeans nam ống đứng, chất liệu denim co giãn.",
        "sale_price": 399000.0,
        "original_price": 499000.0,
        "discount_amount": 100000.0,
        "max_score": 0.85
    },
    {
        "internal_ref_code": "8SK24W003",
        "description_text_full": "Váy liền thân nữ dáng xòe, họa tiết hoa nhí.",
        "sale_price": 350000.0,
        "original_price": 450000.0,
        "discount_amount": 100000.0,
        "max_score": 0.82
    }
]

# --- ENDPOINTS ---

@router.post("/mock/agent/chat", summary="Mock LLM Chat Endpoint (Fast)")
async def mock_chat(req: MockQueryRequest):
    """
    Giả lập phản hồi của Chatbot (bỏ qua LLM & DB thật).
    Dùng để test tải (Load Test) xem server chịu được bao nhiêu request/s.
    """
    # Simulate slight processing time (10ms - 50ms)
    time.sleep(random.uniform(0.01, 0.05))
    
    return {
        "status": "success",
        "user_query": req.user_query,
        "ai_response": {
            "content": f"Đây là câu trả lời giả lập cho: '{req.user_query}'. Hệ thống đang hoạt động tốt!",
            "role": "assistant"
        },
        "product_ids": ["8TE24W001", "8JE24W002"],
        "processing_time": "0.03s"
    }

@router.post("/mock/db/search", summary="Mock StarRocks Vector Search")
async def mock_db_search(req: MockDBRequest):
    """
    Giả lập truy vấn Vector Database (StarRocks).
    Input: Vector embedding (nếu có).
    Output: Danh sách sản phẩm mock giống cấu trúc SQL thật.
    """
    # Simulate DB Latency (e.g., 50ms - 100ms)
    time.sleep(random.uniform(0.05, 0.1))
    
    # Hardcoded vector from user request (Just for checking format)
    # [0.01643567718565464, -0.0008633101242594421, ...] 
    
    return {
        "status": "success",
        "total_hits": 1340,
        "results": MOCK_PRODUCTS, # Return existing mock data
        "query_vector_dim": len(req.vector) if req.vector else 0,
        "note": "This is MOCK data from memory, no actual DB connection."
    }
