"""
FastAPI main application - Contract AI Service

Architecture:
- REST API Routes: FastAPI routers for HTTP endpoints
- SSE (Server-Sent Events): Real-time streaming for AI Contract and AI Legal

Modules:
- ai_contract: Contract generation and composition (SSE streaming)
- ai_legal: Legal Q&A and search (SSE streaming)
- conversation: Conversation history management
"""

import logging
import os

# Configure Logging
logging.basicConfig(
    level=logging.INFO, format="%(asctime)s [%(levelname)s] %(name)s: %(message)s", handlers=[logging.StreamHandler()]
)
logger = logging.getLogger(__name__)


import uvicorn
from fastapi import FastAPI
from fastapi.middleware.cors import CORSMiddleware
from fastapi.staticfiles import StaticFiles  # Import StaticFiles

# Updated APIs
from api.chatbot_route import router as chatbot_router
from api.conservation_route import router as conservation_router
from config import PORT

app = FastAPI(
    title="Contract AI Service",
    description="API for Contract AI Service",
    version="1.0.0",
)

# Clerk Auth (comment out if testing locally without valid token might be needed,
# but user environment has it enabled. Static files usually should be public or protected.
# For now, we mount static files BEFORE auth if we want them public, or just let auth handle it.
# Usually static files for testing are convenient to be public
# or we just need to authenticate via header in fetch.
# But for simple HTML test page, often it is easier if it is open.
# However, app.add_middleware applies to everything.
# Let's assume ClerkAuthMiddleware allows public paths or we just login.
# IMPORTANT: If ClerkAuthMiddleware blocks everything, the static page won't load easily without token.
# But user just asked to mount it.
# app.add_middleware(ClerkAuthMiddleware)
print("✅ Clerk Authentication middleware DISABLED (for testing)")

# Add CORS middleware
app.add_middleware(
    CORSMiddleware,
    allow_origins=["*"],
    allow_credentials=True,
    allow_methods=["*"],
    allow_headers=["*"],
)

# ========== REST API Routes ==========
# Conversation History (Mới)
app.include_router(conservation_router)

# Chatbot Agent (Mới)
app.include_router(chatbot_router, prefix="/api/agent")

# Mount this LAST to avoid conflicts with API routes
try:
    static_dir = os.path.join(os.path.dirname(__file__), "static")
    if not os.path.exists(static_dir):
        os.makedirs(static_dir)
    app.mount("/static", StaticFiles(directory=static_dir, html=True), name="static")
    print(f"✅ Static files mounted at /static (Dir: {static_dir})")
except Exception as e:
    print(f"⚠️ Failed to mount static files: {e}")

@app.get("/")
async def root():
    return {"message": "Contract AI Service is running!", "status": "healthy"}


if __name__ == "__main__":
    print("=" * 60)
    print("🚀 Contract AI Service Starting...")
    print("=" * 60)
    print(f"📡 REST API: http://localhost:{PORT}")
    print(f"📡 SSE Streaming: http://localhost:{PORT}/api/ai-contract/chat")
    print(f"📡 Test Chatbot: http://localhost:{PORT}/static/index.html")
    print(f"📚 API Docs: http://localhost:{PORT}/docs")
    print("=" * 60)

    # ENABLE_RELOAD = os.getenv("ENABLE_RELOAD", "false").lower() in ("true", "1", "yes")
    ENABLE_RELOAD = False  # Tạm thời tắt reload để kiểm tra độ ổn định
    print(f"⚠️  Hot reload: {ENABLE_RELOAD}")

    reload_dirs = [
        "common",
        "api",  # Watch api folder
        "agent",  # Watch agent folder
    ]

    uvicorn.run(
        "server:app",
        host="0.0.0.0",
        port=PORT,
        reload=ENABLE_RELOAD,
        reload_dirs=reload_dirs,
        log_level="info",
    )
