"""
CANIFA Data Retrieval Tool - Tối giản cho Agentic Workflow.
Hỗ trợ Hybrid Search: Semantic (Vector) + Metadata Filter.
"""

import asyncio
import json
import logging
import time
from decimal import Decimal

from langchain_core.tools import tool
from pydantic import BaseModel, Field

from agent.tools.product_search_helpers import build_starrocks_query
from common.starrocks_connection import get_db_connection

# from langsmith import traceable

logger = logging.getLogger(__name__)


class DecimalEncoder(json.JSONEncoder):
    """Xử lý kiểu Decimal từ Database khi convert sang JSON."""

    def default(self, obj):
        if isinstance(obj, Decimal):
            return float(obj)
        return super().default(obj)


class SearchItem(BaseModel):
    """Cấu trúc một mục tìm kiếm đơn lẻ trong Multi-Search."""

    query: str = Field(
        ...,
        description="MÔ TẢ sản phẩm chi tiết (KHÔNG phải câu hỏi thô). VD: 'Áo thun nam cotton basic trẻ trung, phù hợp đi chơi hàng ngày'. Bao gồm: tên SP, chất liệu, phong cách, giới tính (nếu có), màu sắc (nếu có), dịp sử dụng.",
    )
    magento_ref_code: str | None = Field(
        ..., description="Mã sản phẩm hoặc SKU (Ví dụ: 8TS24W001). CHỈ điền khi khách hỏi mã code cụ thể."
    )
    price_min: float | None = Field(..., description="Giá thấp nhất (VD: 100000)")
    price_max: float | None = Field(..., description="Giá cao nhất (VD: 500000)")
    action: str = Field(..., description="Hành động: 'search' (tìm kiếm) hoặc 'visual_search' (phân tích ảnh)")


class MultiSearchParams(BaseModel):
    """Tham số cho Parallel Multi-Search."""

    searches: list[SearchItem] = Field(..., description="Danh sách các truy vấn tìm kiếm chạy song song")


@tool(args_schema=MultiSearchParams)
# @traceable(run_type="tool", name="data_retrieval_tool")
async def data_retrieval_tool(searches: list[SearchItem]) -> str:
    """
    Siêu công cụ tìm kiếm sản phẩm CANIFA - Hỗ trợ Parallel Multi-Search (Chạy song song nhiều query).

    💡 ĐIỂM ĐẶC BIỆT:
    Công cụ này cho phép thực hiện NHIỀU truy vấn tìm kiếm CÙNG LÚC.
    Hãy dùng nó khi cần SO SÁNH sản phẩm hoặc tìm trọn bộ OUTFIT (mix & match).

    ⚠️ QUAN TRỌNG - CÁCH DÙNG:

    1️⃣ DÙNG 'query' (HyDE Semantic Search - BẮT BUỘC):
       - MÔ TẢ chi tiết sản phẩm: tên, chất liệu, giới tính, màu sắc, phong cách, dịp sử dụng
       - VD: "Áo thun nam cotton màu đỏ basic trẻ trung, phù hợp đi chơi hàng ngày"
       - **KHÔNG viết câu hỏi thô**, KHÔNG ném giá vào query

    2️⃣ DÙNG 'magento_ref_code':
       - CHỈ khi khách hỏi mã sản phẩm cụ thể: "Mã 8TS24W001", "SP 6OT25W020"

    3️⃣ DÙNG 'price_min' / 'price_max':
       - Khi khách nói về giá: "dưới 500k", "từ 200k đến 400k"

    📝 VÍ DỤ (Single Search):
       - searches=[{"query": "Áo polo nam cotton cao cấp lịch sự công sở", "price_max": 400000}]
       - searches=[{"magento_ref_code": "8TS24W001"}]

    🚀 VÍ DỤ (Multi-Search):
       - So sánh: searches=[
           {"query": "Áo thun nam cotton màu đen basic casual đi chơi", "price_max": 500000},
           {"query": "Áo sơ mi nam màu trắng lịch sự công sở", "price_max": 500000}
         ]
       - Phối đồ: searches=[
           {"query": "Quần jean nam slim fit năng động"},
           {"query": "Áo khoác nam thể thao trẻ trung"}
         ]
    """
    logger.info("🔧 [DEBUG] data_retrieval_tool STARTED")
    try:
        logger.info("🔧 [DEBUG] Getting DB connection (singleton)")
        db = get_db_connection()
        logger.info("🔧 [DEBUG] DB connection retrieved successfully")

        # 0. Log input parameters (Đúng ý bro)
        logger.info(f"📥 [Tool Input] data_retrieval_tool received {len(searches)} items:")
        for idx, item in enumerate(searches):
            logger.info(f"   🔹 Item [{idx}]: {item.dict(exclude_none=True)}")

        # 1. Tạo tasks chạy song song (Parallel)
        logger.info("🔧 [DEBUG] Creating parallel tasks")
        tasks = []
        for item in searches:
            tasks.append(_execute_single_search(db, item))

        logger.info(f"🚀 [Parallel Search] Executing {len(searches)} queries simultaneously...")
        logger.info("🔧 [DEBUG] About to call asyncio.gather()")
        results = await asyncio.gather(*tasks)
        logger.info(f"🔧 [DEBUG] asyncio.gather() completed with {len(results)} results")

        # 2. Tổng hợp kết quả
        combined_results = []
        for i, products in enumerate(results):
            combined_results.append(
                {
                    "search_index": i,
                    "search_criteria": searches[i].dict(exclude_none=True),
                    "count": len(products),
                    "products": products,
                }
            )

        return json.dumps({"status": "success", "results": combined_results}, ensure_ascii=False, cls=DecimalEncoder)

    except Exception as e:
        logger.error(f"Error in Multi-Search data_retrieval_tool: {e}")
        return json.dumps({"status": "error", "message": str(e)})


async def _execute_single_search(db, item: SearchItem) -> list[dict]:
    """Thực thi một search query đơn lẻ (Async)."""
    try:
        logger.info(f"🔧 [DEBUG] _execute_single_search STARTED for query: {item.query[:50] if item.query else 'None'}")

        # ⏱️ Timer: Build query (bao gồm embedding nếu có)
        query_build_start = time.time()
        logger.info("🔧 [DEBUG] Calling build_starrocks_query()")
        sql = await build_starrocks_query(item)
        query_build_time = (time.time() - query_build_start) * 1000  # Convert to ms
        logger.info(f"🔧 [DEBUG] SQL query built, length: {len(sql)}")
        logger.info(f"⏱️ [TIMER] Query Build Time (bao gồm embedding): {query_build_time:.2f}ms")

        # ⏱️ Timer: Execute DB query
        db_start = time.time()
        logger.info("🔧 [DEBUG] Calling db.execute_query_async()")
        products = await db.execute_query_async(sql)
        db_time = (time.time() - db_start) * 1000  # Convert to ms
        logger.info(f"🔧 [DEBUG] Query executed, got {len(products)} products")
        logger.info(f"⏱️ [TIMER] DB Query Execution Time: {db_time:.2f}ms")
        logger.info(f"⏱️ [TIMER] Total Time (Build + DB): {query_build_time + db_time:.2f}ms")

        # Ghi log DB Preview (Kết quả thực tế) vào Background Task
        search_label = item.magento_ref_code if item.magento_ref_code else item.query
        # asyncio.create_task(save_preview_to_log(search_label, products))

        return _format_product_results(products)
    except Exception as e:
        logger.error(f"Single search error for item {item}: {e}")
        return []


def _format_product_results(products: list[dict]) -> list[dict]:
    """Lọc và format kết quả trả về cho Agent."""
    allowed_fields = {
        "internal_ref_code",
        "description_text_full",
        "sale_price",
        "original_price",
        "discount_amount",
        "max_score",
    }
    return [{k: v for k, v in p.items() if k in allowed_fields} for p in products[:5]]
