from fastapi import APIRouter, HTTPException
from typing import List, Dict, Any, Optional
from pydantic import BaseModel
import logging

from common.conversation_manager import get_conversation_manager

router = APIRouter(tags=["Conservation"])
logger = logging.getLogger(__name__)

class ChatMessage(BaseModel):
    id: int
    user_id: str | None = None # Optional usually not needed in list but good for consistency
    message: str
    is_human: bool
    timestamp: Any

class ChatHistoryResponse(BaseModel):
    data: List[Dict[str, Any]]
    next_cursor: Optional[int] = None

@router.get("/history/{user_id}", summary="Get Chat History by User ID", response_model=ChatHistoryResponse)
async def get_chat_history(user_id: str, limit: Optional[int] = 20, before_id: Optional[int] = None):
    """
    Lấy lịch sử chat của user từ Postgres database.
    Trả về object chứa `data` (list messages) và `next_cursor` để dùng cho trang tiếp theo.
    """
    try:
        # Sử dụng ConversationManager Singleton
        manager = get_conversation_manager()
        
        # Lấy history từ DB với pagination
        history = manager.get_chat_history(user_id, limit=limit, before_id=before_id)
        
        next_cursor = None
        if history and len(history) > 0:
            # Cursor cho trang tiếp theo chính là ID của tin nhắn cuối cùng (cũ nhất trong batch này)
            next_cursor = history[-1]['id']

        return {
            "data": history,
            "next_cursor": next_cursor
        }
    except Exception as e:
        logger.error(f"Error fetching chat history for user {user_id}: {e}")
        raise HTTPException(status_code=500, detail=str(e))
