"""
Fashion Q&A Agent Router
FastAPI endpoints cho Fashion Q&A Agent service.
Router chỉ chứa định nghĩa API, logic nằm ở controller.
"""

import json
import logging
from collections.abc import AsyncGenerator

from fastapi import APIRouter, HTTPException, Request
from fastapi.responses import StreamingResponse

from agent.controller import chat_controller
from agent.models import QueryRequest
from config import DEFAULT_MODEL

logger = logging.getLogger(__name__)
router = APIRouter()


@router.post("/chat", summary="Fashion Q&A Chat (Non-streaming)")
async def fashion_qa_chat(req: QueryRequest, request: Request):
    """
    Endpoint chat không stream - trả về response JSON đầy đủ một lần.
    """
    # Trích xuất user_id từ request (auth middleware)
    user_id = getattr(request.state, "user_id", None) or req.user_id or "default_user"
    
    logger.info(f"📥 [Incoming Query - NonStream] User: {user_id} | Query: {req.query}")

    try:
        # Gọi controller để xử lý logic và nhận generator stream
        # Note: Vì chat_controller có decorator @observe(), cần await để unwrap
        generator: AsyncGenerator[str, None] = chat_controller(
            query=req.query,
            user_id=user_id,
            model_name=DEFAULT_MODEL,
            conversation_id=req.conversation_id,
            images=req.images,
        )

        # Collect toàn bộ events từ generator
        final_response = None
        async for chunk in generator:
            # Parse SSE data format
            if chunk.startswith("data: "):
                data_str = chunk[6:].strip()
                if data_str != "[DONE]":
                    final_response = json.loads(data_str)

        # Trả về response cuối cùng
        if final_response and "messages" in final_response:
            last_message = final_response["messages"][-1]
            response_text = last_message.get("content", "") if isinstance(last_message, dict) else str(last_message)
            
            logger.info(f"📤 [Outgoing Response - NonStream] User: {user_id} | Response: {response_text}")

            return {
                "status": "success",
                "response": response_text,
                "conversation_id": req.conversation_id,
            }

        return {"status": "error", "message": "No response generated"}

    except Exception as e:
        logger.error(f"Error in fashion_qa_chat: {e}", exc_info=True)
        raise HTTPException(status_code=500, detail=str(e)) from e


# ====================== FASHION Q&A CHAT API ======================
@router.post("/stream/chat", summary="Fashion Q&A Chat with Streaming Response")
async def fashion_qa_chat_stream(req: QueryRequest, request: Request):
    """
    Endpoint duy nhất cho việc chat với Fashion Agent.
    """
    # Trích xuất user_id từ request (auth middleware)
    user_id = getattr(request.state, "user_id", None) or req.user_id or "default_user"
    
    logger.info(f"📥 [Incoming Query] User: {user_id} | Query: {req.query}")

    try:
        # Gọi controller để xử lý logic và nhận generator stream
        # Note: Vì chat_controller có decorator @observe(), cần await để unwrap
        generator: AsyncGenerator[str, None] = chat_controller(
            query=req.query,
            user_id=user_id,
            model_name=DEFAULT_MODEL,
            conversation_id=req.conversation_id,
            images=req.images,
        )

        async def logging_generator(gen: AsyncGenerator[str, None]):
            full_response_log = ""
            first_chunk = True
            try:
                async for chunk in gen:
                    if first_chunk:
                        logger.info("🚀 [Stream Started] First chunk received")
                        first_chunk = False
                    
                    full_response_log += chunk
                    yield chunk
            except Exception as e:
                logger.error(f"❌ [Stream Error] {e}")
                yield f"data: {json.dumps({'error': str(e)})}\n\n"
            
            logger.info(f"📤 [Outgoing Response Stream Finished] Total Chunks Length: {len(full_response_log)}")

        return StreamingResponse(
            logging_generator(generator),
            media_type="text/event-stream",
            headers={
                "Cache-Control": "no-cache",
                "Connection": "keep-alive",
                "X-Accel-Buffering": "no",
            },
        )
    except Exception as e:
        logger.error(f"Error in fashion_qa_chat: {e}", exc_info=True)
        raise HTTPException(status_code=500, detail=str(e)) from e
