"""
Fashion Q&A Agent Controller
Điều phối luồng chạy của Agent, tích hợp ConversationManager (Postgres Memory).
Switched to LangSmith for tracing (configured via environment variables).
"""

import json
import logging
import uuid
from collections.abc import AsyncGenerator

from langchain_core.messages import AIMessage, HumanMessage
from langchain_core.runnables import RunnableConfig

from common.llm_factory import create_llm
from common.conversation_manager import get_conversation_manager, ConversationManager
from config import DEFAULT_MODEL

from .graph import build_graph
from .models import AgentState, get_config
from .tools.get_tools import get_all_tools

logger = logging.getLogger(__name__)

async def chat_controller(
    query: str, user_id: str, model_name: str = DEFAULT_MODEL, conversation_id: str | None = None, images: list[str] | None = None
) -> AsyncGenerator[str, None]:
    # 1. Khởi tạo & Chuẩn bị (Dependency Injection)
    logger.info(f"▶️ Starting chat_controller with model: {model_name} for user: {user_id}")
    config = get_config()
    config.model_name = model_name

    # Khởi tạo resources - Factory sẽ tự động chọn provider dựa trên tên model
    llm = create_llm(model_name=model_name, streaming=True)
    tools = get_all_tools()

    graph = build_graph(config, llm=llm, tools=tools)
    
    # Init ConversationManager (Singleton)
    memory = get_conversation_manager()
    
    actual_conv_id = conversation_id or str(uuid.uuid4())

    # LOAD HISTORY & Prepare State
    # Get history from Postgres (returns list of dicts)
    history_dicts = memory.get_chat_history(user_id, limit=10)
    
    # Convert to BaseMessage objects
    history = []
    for h in reversed(history_dicts): # API returns desc, we want chronological for context
        if h['is_human']:
            history.append(HumanMessage(content=h['message']))
        else:
            history.append(AIMessage(content=h['message']))

    current_human_msg = HumanMessage(content=query)

    initial_state, exec_config = _prepare_execution_context(
        query=query, user_id=user_id, actual_conv_id=actual_conv_id, history=history, images=images
    )

    final_ai_message = None

    # 3. Stream Engine
    try:
        async for event in graph.astream(initial_state, config=exec_config, stream_mode="values"):
            final_ai_message = _extract_last_ai_message(event) or final_ai_message
            
            # Serialize messages to dicts to avoid "content='...'" string representation
            if "messages" in event:
                event["messages"] = [
                    m.dict() if hasattr(m, "dict") else m 
                    for m in event["messages"]
                ]
            
            yield f"data: {json.dumps(event, default=str, ensure_ascii=False)}\n\n"

        # 4. Hậu xử lý (Lưu DB)
        _handle_post_chat(
            memory=memory,
            user_id=user_id,
            human_query=query,
            ai_msg=final_ai_message,
        )
        yield "data: [DONE]\n\n"

    except Exception as e:
        logger.error(f"💥 Stream error: {e}", exc_info=True)
        yield f"data: {json.dumps({'error': str(e)})}\n\n"
    finally:
        logger.info(f"✅ Request completed for conversation {actual_conv_id}")

def _prepare_execution_context(query: str, user_id: str, actual_conv_id: str, history: list, images: list | None):
    """Tách logic chuẩn bị state và config để giảm độ phức tạp."""
    initial_state: AgentState = {
        "messages": [HumanMessage(content=query)],
        "history": history,
        "user_id": user_id,
        "images": [],
        "thread_id": actual_conv_id,
        "image_analysis": None,
    }

    run_id = str(uuid.uuid4())
    # Metadata for LangSmith
    metadata = {
        "conversation_id": actual_conv_id,
        "user_id": user_id,
        "run_id": run_id
    }
    
    exec_config = RunnableConfig(
        configurable={
            "conversation_id": actual_conv_id,
            "user_id": user_id,
            "transient_images": images or [],
            "run_id": run_id,
        },
        run_id=run_id,
        metadata=metadata, # Attach metadata for LangSmith
    )
    return initial_state, exec_config


def _extract_last_ai_message(event: dict) -> AIMessage | None:
    """Trích xuất tin nhắn AI cuối cùng từ event stream."""
    if event.get("messages"):
        last_msg = event["messages"][-1]
        if isinstance(last_msg, AIMessage):
            return last_msg
    return None


def _handle_post_chat(memory: ConversationManager, user_id: str, human_query: str, ai_msg: AIMessage | None):
    """Xử lý lưu history sau khi kết thúc stream. LangSmith tự động trace nên không cần update thủ công."""
    if ai_msg:
        # Save User Message
        memory.save_message(user_id, human_query, True)
        # Save AI Message
        memory.save_message(user_id, ai_msg.content, False)
        
        logger.info(f"💾 Saved conversation for user {user_id} to Postgres")
