import asyncio
import os
import platform

if platform.system() == "Windows":
    print("🔧 Windows detected: Applying SelectorEventLoopPolicy globally...")
    asyncio.set_event_loop_policy(asyncio.WindowsSelectorEventLoopPolicy())

import logging

import uvicorn
from fastapi import FastAPI
from fastapi.staticfiles import StaticFiles

from api.chatbot_route import router as chatbot_router
from api.conservation_route import router as conservation_router
from api.prompt_route import router as prompt_router
from common.cache import redis_cache
from common.langfuse_client import get_langfuse_client
from common.middleware import middleware_manager
from config import PORT

# Configure Logging
logging.basicConfig(
    level=logging.INFO,
    format="%(asctime)s [%(levelname)s] %(name)s: %(message)s",
    handlers=[logging.StreamHandler()],
)

logger = logging.getLogger(__name__)

langfuse_client = get_langfuse_client()
if langfuse_client:
    logger.info("✅ Langfuse client ready (lazy loading)")
else:
    logger.warning("⚠️ Langfuse client not available (missing keys or disabled)")

app = FastAPI(
    title="Contract AI Service",
    description="API for Contract AI Service",
    version="1.0.0",
)


# =============================================================================
# STARTUP EVENT - Initialize Redis Cache
# =============================================================================
@app.on_event("startup")
async def startup_event():
    """Initialize Redis cache on startup."""
    await redis_cache.initialize()
    logger.info("✅ Redis cache initialized for message limit")


# =============================================================================
# MIDDLEWARE SETUP - Gom Auth + RateLimit + CORS vào một chỗ
# =============================================================================
middleware_manager.setup(
    app,
    enable_auth=True,        # 👈 Bật lại Auth để test logic Guest/User
    enable_rate_limit=True,   # 👈 Bật lại SlowAPI theo yêu cầu
    enable_cors=True,         # 👈 Bật CORS
    cors_origins=["*"],       # 👈 Trong production nên limit origins
)

app.include_router(conservation_router)
app.include_router(chatbot_router)
app.include_router(prompt_router)


# --- MOCK API FOR LOAD TESTING ---
try:
    from api.mock_api_route import router as mock_router

    app.include_router(mock_router)
    print("✅ Mock API Router mounted at /mock")
except ImportError:
    print("⚠️ Mock Router not found, skipping...")

# ==========================================
# 🟢 ĐOẠN MOUNT STATIC HTML CỦA BRO ĐÂY 🟢
# ==========================================
try:
    static_dir = os.path.join(os.path.dirname(__file__), "static")
    if not os.path.exists(static_dir):
        os.makedirs(static_dir)
    # Mount thư mục static để chạy file index.html
    app.mount("/static", StaticFiles(directory=static_dir, html=True), name="static")
    print(f"✅ Static files mounted at /static (Dir: {static_dir})")
except Exception as e:
    print(f"⚠️ Failed to mount static files: {e}")


from fastapi.responses import RedirectResponse


@app.get("/")
async def root():
    return RedirectResponse(url="/static/index.html")


if __name__ == "__main__":
    print("=" * 60)
    print("🚀 Contract AI Service Starting...")
    print("=" * 60)
    print(f"📡 REST API: http://localhost:{PORT}")
    print(f"📡 Test Chatbot: http://localhost:{PORT}/static/index.html")
    print(f"📚 API Docs: http://localhost:{PORT}/docs")
    print("=" * 60)

    ENABLE_RELOAD = False
    print(f"⚠️  Hot reload: {ENABLE_RELOAD}")

    reload_dirs = ["common", "api", "agent"]

    if ENABLE_RELOAD:
        os.environ["PYTHONUNBUFFERED"] = "1"

    uvicorn.run(
        "server:app",
        host="0.0.0.0",
        port=PORT,
        reload=ENABLE_RELOAD,
        reload_dirs=reload_dirs,
        log_level="info",
    )
