
import logging
from datetime import datetime
from common.cache import redis_cache

logger = logging.getLogger(__name__)

class ResetLimitService:
    def __init__(self, limit: int = 5):
        self.limit = limit
        self.expiration_seconds = 86400  # 1 day

    async def check_limit(self, identity_key: str) -> tuple[bool, int, int]:
        """
        Check if user can reset chat.
        Returns (can_reset, current_usage, remaining)
        """
        redis_client = redis_cache.get_client()
        if not redis_client:
            # Fallback if Redis is down: allow reset
            return True, 0, self.limit

        today = datetime.now().strftime("%Y-%m-%d")
        key = f"reset_limit:{identity_key}:{today}"
        
        try:
            count = await redis_client.get(key)
            if count is None:
                return True, 0, self.limit
            
            current_usage = int(count)
            remaining = self.limit - current_usage
            
            if current_usage >= self.limit:
                return False, current_usage, 0
            
            return True, current_usage, remaining
        except Exception as e:
            logger.error(f"Error checking reset limit: {e}")
            return True, 0, self.limit

    async def increment(self, identity_key: str):
        redis_client = redis_cache.get_client()
        if not redis_client:
            return

        today = datetime.now().strftime("%Y-%m-%d")
        key = f"reset_limit:{identity_key}:{today}"
        
        try:
            pipe = redis_client.pipeline()
            pipe.incr(key)
            pipe.expire(key, self.expiration_seconds)
            await pipe.execute()
        except Exception as e:
            logger.error(f"Error incrementing reset limit: {e}")

reset_limit_service = ResetLimitService(limit=5)
