"""
Chat History API Routes
- GET /api/history/{identity_key} - Lấy lịch sử chat (có product_ids)
- DELETE /api/history/{identity_key} - Xóa lịch sử chat

Note: identity_key có thể là device_id (guest) hoặc user_id (đã login)
"""

import logging
from typing import Any

from fastapi import APIRouter, HTTPException
from pydantic import BaseModel

from common.conversation_manager import get_conversation_manager

router = APIRouter(tags=["Chat History"])
logger = logging.getLogger(__name__)


class ChatHistoryResponse(BaseModel):
    data: list[dict[str, Any]]
    next_cursor: int | None = None


class ClearHistoryResponse(BaseModel):
    success: bool
    message: str


@router.get("/api/history/{identity_key}", summary="Get Chat History", response_model=ChatHistoryResponse)
async def get_chat_history(identity_key: str, limit: int | None = 50, before_id: int | None = None):
    """
    Lấy lịch sử chat theo identity_key.
    
    identity_key:
    - Guest: device_id
    - User đã login: user_id (customer_id từ Canifa)
    
    Response bao gồm:
    - message: Nội dung tin nhắn
    - is_human: True nếu là user, False nếu là AI
    - product_ids: List sản phẩm liên quan (chỉ có với AI messages)
    - timestamp: Thời gian
    - id: ID tin nhắn (dùng cho pagination)
    """
    try:
        manager = await get_conversation_manager()
        history = await manager.get_chat_history(identity_key, limit=limit, before_id=before_id)

        next_cursor = None
        if history and len(history) > 0:
            next_cursor = history[-1]["id"]

        return {"data": history, "next_cursor": next_cursor}
    except Exception as e:
        logger.error(f"Error fetching chat history for {identity_key}: {e}")
        raise HTTPException(status_code=500, detail="Failed to fetch chat history")


@router.delete("/api/history/{identity_key}", summary="Clear Chat History", response_model=ClearHistoryResponse)
async def clear_chat_history(identity_key: str):
    """
    Xóa toàn bộ lịch sử chat theo identity_key.
    """
    try:
        manager = await get_conversation_manager()
        await manager.clear_history(identity_key)
        logger.info(f"✅ Cleared chat history for {identity_key}")
        return {"success": True, "message": f"Đã xóa lịch sử chat của {identity_key}"}
    except Exception as e:
        logger.error(f"Error clearing chat history for {identity_key}: {e}")
        raise HTTPException(status_code=500, detail="Failed to clear chat history")

