"""
Fashion Q&A Agent Router
FastAPI endpoints cho Fashion Q&A Agent service.
Router chỉ chứa định nghĩa API, logic nằm ở controller.

Note: Rate limit check đã được xử lý trong middleware (CanifaAuthMiddleware)
"""

import logging

from fastapi import APIRouter, BackgroundTasks, HTTPException, Request
from fastapi.responses import JSONResponse

from agent.controller import chat_controller
from agent.models import QueryRequest
from common.message_limit import message_limit_service
from common.rate_limit import rate_limit_service
from config import DEFAULT_MODEL

logger = logging.getLogger(__name__)
router = APIRouter()

@router.post("/api/agent/chat", summary="Fashion Q&A Chat (Non-streaming)")
@rate_limit_service.limiter.limit("50/minute")
async def fashion_qa_chat(request: Request, req: QueryRequest, background_tasks: BackgroundTasks):
    """
    Endpoint chat không stream - trả về response JSON đầy đủ một lần.
    
    Note: Rate limit đã được check trong middleware.
    """
    # 1. Lấy user identity từ Middleware (request.state)
    # Logic: Login -> User ID | Guest -> Device ID
    user_id = getattr(request.state, "user_id", None)
    device_id = getattr(request.state, "device_id", "unknown")
    is_authenticated = getattr(request.state, "is_authenticated", False)
    
    # Định danh duy nhất cho Request này (Log, History, Rate Limit, Langfuse)
    identity_id = user_id if is_authenticated else device_id
    
    # Rate limit đã check trong middleware, lấy limit_info từ request.state
    limit_info = getattr(request.state, 'limit_info', None)

    logger.info(f"📥 [Incoming Query - NonStream] User: {identity_id} | Query: {req.user_query}")

    try:
        # Gọi controller để xử lý logic (Non-streaming)
        result = await chat_controller(
            query=req.user_query,
            user_id=str(identity_id), # Langfuse User ID
            background_tasks=background_tasks,
            model_name=DEFAULT_MODEL,
            images=req.images,
            identity_key=str(identity_id),  # Key lưu history
        )

        # Log chi tiết response
        logger.info(f"📤 [Outgoing Response - NonStream] User: {identity_id}")
        logger.info(f"💬 AI Response: {result['ai_response']}")
        logger.info(f"🛍️ Product IDs: {result.get('product_ids', [])}")

        # Increment message count SAU KHI chat thành công
        usage_info = await message_limit_service.increment(
            identity_key=identity_id,
            is_authenticated=is_authenticated,
        )

        return {
            "status": "success",
            "ai_response": result["ai_response"],
            "product_ids": result.get("product_ids", []),
            "limit_info": {
                "limit": usage_info["limit"],
                "used": usage_info["used"],
                "remaining": usage_info["remaining"],
            },
        }
    except Exception as e:
        logger.error(f"Error in fashion_qa_chat: {e}", exc_info=True)
        # Trả về lỗi dạng JSON chuẩn với error_code="SYSTEM_ERROR"
        return JSONResponse(
            status_code=500,
            content={
                "status": "error",
                "error_code": "SYSTEM_ERROR",
                "message": "Oops 😥 Hiện Canifa-AI chưa thể xử lý yêu cầu của bạn ngay lúc này, vui lòng quay lại trong giây lát."
            }
        )

