"""
Fashion Q&A Agent Graph
LangGraph workflow với clean architecture.
Tất cả resources (LLM, Tools) khởi tạo trong __init__.
Sử dụng ConversationManager (Postgres) để lưu history thay vì checkpoint.
"""

import asyncio
import logging
from typing import Any

from langchain_core.language_models import BaseChatModel
from langchain_core.prompts import ChatPromptTemplate, MessagesPlaceholder
from langchain_core.runnables import RunnableConfig
# from langgraph.cache.memory import InMemoryCache  # DISABLED FOR DEBUG
from langgraph.graph import END, StateGraph
from langgraph.prebuilt import ToolNode
# from langgraph.types import CachePolicy  # DISABLED FOR DEBUG

from common.llm_factory import create_llm

from .models import AgentConfig, AgentState, get_config
from .prompt import get_system_prompt
from .tools.get_tools import get_all_tools, get_collection_tools

logger = logging.getLogger(__name__)


class CANIFAGraph:
    """
    Fashion Q&A Agent Graph Manager.
    """

    def __init__(
        self,
        config: AgentConfig | None = None,
        llm: BaseChatModel | None = None,
        tools: list | None = None,
    ):
        self.config = config or get_config()
        self._compiled_graph: Any | None = None
        self.llm: BaseChatModel = llm or create_llm(
            model_name=self.config.model_name, api_key=self.config.openai_api_key, streaming=True
        )
        self.all_tools = tools or get_all_tools()
        self.collection_tools = get_collection_tools()  # Vẫn lấy list name để routing
        self.retrieval_tools = self.all_tools

        self.llm_with_tools = self.llm.bind_tools(self.all_tools, strict=True)
        self.system_prompt = get_system_prompt()
        self.prompt_template = ChatPromptTemplate.from_messages(
            [
                ("system", self.system_prompt),
                MessagesPlaceholder(variable_name="history"),
                MessagesPlaceholder(variable_name="user_query"),
                MessagesPlaceholder(variable_name="messages"),
            ]
        )
        self.chain = self.prompt_template | self.llm_with_tools
        # self.cache = InMemoryCache()  # DISABLED FOR DEBUG

    async def _agent_node(self, state: AgentState, config: RunnableConfig) -> dict:
        """Agent node - Chỉ việc đổ dữ liệu riêng vào khuôn đã có sẵn."""
        logger.info("🔧 [DEBUG] _agent_node CALLED!")
        messages = state.get("messages", [])
        history = state.get("history", [])
        user_query = state.get("user_query")
        logger.info(f"🔧 [DEBUG] _agent_node processing {len(messages)} messages")

        transient_images = config.get("configurable", {}).get("transient_images", [])
        if transient_images and messages:
            pass
        # Invoke chain with user_query, history, and messages
        logger.info("🔧 [DEBUG] About to invoke chain with 60s timeout")
        try:
            response = await asyncio.wait_for(
                self.chain.ainvoke({
                    "user_query": [user_query] if user_query else [],
                    "history": history,
                    "messages": messages
                }),
                timeout=60.0
            )
            logger.info("🔧 [DEBUG] Chain invoked successfully")
            return {"messages": [response], "ai_response": response}
        except asyncio.TimeoutError:
            logger.error("❌ Chain invoke TIMEOUT after 60s!")
            raise TimeoutError("LLM chain invoke timed out after 60 seconds")


    def _should_continue(self, state: AgentState) -> str:
        """Routing: tool nodes hoặc end."""
        last_message = state["messages"][-1]

        if not hasattr(last_message, "tool_calls") or not last_message.tool_calls:
            logger.info("🏁 Agent finished")
            return "end"

        tool_names = [tc["name"] for tc in last_message.tool_calls]
        collection_names = [t.name for t in self.collection_tools]

        if any(name in collection_names for name in tool_names):
            logger.info(f"🔄 → collect_tools: {tool_names}")
            return "collect_tools"

        logger.info(f"🔄 → retrieve_tools: {tool_names}")
        return "retrieve_tools"

    def build(self) -> Any:
        """Build và compile LangGraph workflow."""
        if self._compiled_graph is not None:
            return self._compiled_graph
        workflow = StateGraph(AgentState)

        # Nodes
        workflow.add_node("agent", self._agent_node)
        workflow.add_node("retrieve_tools", ToolNode(self.retrieval_tools))  # cache_policy DISABLED
        workflow.add_node("collect_tools", ToolNode(self.collection_tools))

        # Edges
        workflow.set_entry_point("agent")
        workflow.add_conditional_edges(
            "agent",
            self._should_continue,
            {"retrieve_tools": "retrieve_tools", "collect_tools": "collect_tools", "end": END},
        )
        workflow.add_edge("retrieve_tools", "agent")
        workflow.add_edge("collect_tools", "agent")

        self._compiled_graph = workflow.compile()  # No Checkpointer, No Cache (DEBUG)
        logger.info("✅ Graph compiled WITHOUT cache (DEBUG MODE)")

        return self._compiled_graph

    @property
    def graph(self) -> Any:
        return self.build()


# --- Singleton & Public API ---
_instance: list[CANIFAGraph | None] = [None]

def build_graph(config: AgentConfig | None = None, llm: BaseChatModel | None = None, tools: list | None = None) -> Any:
    """Get compiled graph (DISABLED SINGLETON FOR DEBUG)."""
    # ALWAYS create new instance to avoid async state conflicts
    logger.info("🔧 [DEBUG] Building NEW graph instance (singleton disabled)")
    instance = CANIFAGraph(config, llm, tools)
    return instance.build()


def get_graph_manager(
    config: AgentConfig | None = None, llm: BaseChatModel | None = None, tools: list | None = None
) -> CANIFAGraph:
    """Get CANIFAGraph instance."""
    if _instance[0] is None:
        _instance[0] = CANIFAGraph(config, llm, tools)
    return _instance[0]


def reset_graph() -> None:
    """Reset singleton for testing."""
    _instance[0] = None
