import logging

from langchain_core.tools import tool
from pydantic import BaseModel, Field

from common.embedding_service import create_embedding_async
from common.starrocks_connection import get_db_connection

logger = logging.getLogger(__name__)


class KnowledgeSearchInput(BaseModel):
    query: str = Field(
        description="Câu hỏi hoặc nhu cầu tìm kiếm thông tin phi sản phẩm của khách hàng (ví dụ: tìm cửa hàng, hỏi chính sách, tra bảng size...)"
    )


@tool("canifa_knowledge_search", args_schema=KnowledgeSearchInput)
async def canifa_knowledge_search(query: str) -> str:
    """
    Tra cứu TOÀN BỘ thông tin về thương hiệu và dịch vụ của Canifa.
    Sử dụng tool này khi khách hàng hỏi về:

    1. THƯƠNG HIỆU & GIỚI THIỆU: Lịch sử hình thành, giá trị cốt lõi, sứ mệnh.
    2. HỆ THỐNG CỬA HÀNG: Tìm địa chỉ, số điện thoại, giờ mở cửa các cửa hàng tại các tỉnh thành (Hà Nội, HCM, Đà Nẵng, v.v.).
    3. CHÍNH SÁCH BÁN HÀNG: Quy định đổi trả, bảo hành, chính sách vận chuyển, phí ship.
    4. KHÁCH HÀNG THÂN THIẾT (KHTT): Điều kiện đăng ký thành viên, các hạng thẻ (Green, Silver, Gold, Diamond), quyền lợi tích điểm, thẻ quà tặng.
    5. HỖ TRỢ & FAQ: Giải đáp thắc mắc thường gặp, chính sách bảo mật, thông tin liên hệ văn phòng, tuyển dụng.
    6. TRA CỨU SIZE (BẢNG KÍCH CỠ): Hướng dẫn chọn size chuẩn cho nam, nữ, trẻ em dựa trên chiều cao, cân nặng.
    7. GIẢI NGHĨA TỪ VIẾT TẮT: Tự động hiểu các từ viết tắt phổ biến của khách hàng (ví dụ: 'ct' = 'chương trình khuyến mãi/ưu đãi', 'khtt' = 'khách hàng thân thiết', 'store' = 'cửa hàng', 'đc' = 'địa chỉ').

    Ví dụ các câu hỏi phù hợp:
    - 'Bên bạn đang có ct gì không?' (Hiểu là: Chương trình khuyến mãi)
    - 'Canifa ở Cầu Giấy địa chỉ ở đâu?'
    - 'Chính sách đổi trả hàng trong bao nhiêu ngày?'
    - 'Làm sao để lên hạng thẻ Gold?'
    - 'Cho mình xem bảng size áo nam.'
    - 'Phí vận chuyển đi tỉnh là bao nhiêu?'    
    - 'Canifa thành lập năm nào?'
    """
    logger.info(f"🔍 [Semantic Search] Brand Knowledge query: {query}")

    try:
        # 1. Tạo embedding cho câu hỏi (Mặc định 1536 chiều như bro yêu cầu)
        query_vector = await create_embedding_async(query)
        if not query_vector:
            return "Xin lỗi, tôi gặp sự cố khi xử lý thông tin. Vui lòng thử lại sau."

        v_str = "[" + ",".join(str(v) for v in query_vector) + "]"

        # 2. Query StarRocks lấy Top 4 kết quả phù hợp nhất (Không check score)
        sql = f"""
        SELECT 
            content,
            metadata
        FROM shared_source.chatbot_rsa_knowledge
        ORDER BY approx_cosine_similarity(embedding, {v_str}) DESC
        LIMIT 4
        """

        sr = get_db_connection()
        results = await sr.execute_query_async(sql)

        if not results:
            logger.warning(f"⚠️ No knowledge data found in DB for query: {query}")
            return "Hiện tại tôi chưa tìm thấy thông tin chính xác về nội dung này trong hệ thống kiến thức của Canifa. Bạn có thể liên hệ hotline 1800 6061 để được hỗ trợ trực tiếp."

        # 3. Tổng hợp kết quả
        knowledge_texts = []
        for i, res in enumerate(results):
            content = res.get("content", "")
            knowledge_texts.append(content)
            # LOG DỮ LIỆU LẤY ĐƯỢC (Chỉ hiển thị nội dung)
            logger.info(f"📄 [Knowledge Chunk {i + 1}]: {content[:200]}...")

        final_response = "\n\n---\n\n".join(knowledge_texts)
        logger.info(f"✅ Found {len(results)} relevant knowledge chunks.")

        return final_response

    except Exception as e:
        logger.error(f"❌ Error in canifa_knowledge_search: {e}")
        return "Tôi đang gặp khó khăn khi truy cập kho kiến thức. Bạn muốn hỏi về sản phẩm gì khác không?"
