"""
CANIFA Agent với Agno Framework
Thay thế LangGraph bằng Agno
"""

import logging
from typing import TYPE_CHECKING, Any, cast

# Type checking imports (only used for type hints)
if TYPE_CHECKING:
    from agno.agent import Agent as AgentType
    from agno.db.base import BaseDb as BaseDbType
    from agno.models.openai import OpenAIChat as OpenAIChatType
else:
    AgentType = Any  # type: ignore
    BaseDbType = Any  # type: ignore
    OpenAIChatType = Any  # type: ignore

# Runtime imports with fallback
try:
    from agno.agent import Agent
    from agno.db.base import BaseDb
    from agno.models.openai import OpenAIChat
except ImportError:
    # Fallback nếu chưa install agno
    Agent = None
    BaseDb = Any  # type: ignore
    OpenAIChat = None

from common.conversation_manager import get_conversation_manager
from config import DEFAULT_MODEL, OPENAI_API_KEY

from .prompt import get_system_prompt
from .tools.agno_tools import get_agno_tools

logger = logging.getLogger(__name__)


def create_agno_model(model_name: str = DEFAULT_MODEL, json_mode: bool = False):
    """
    Tạo Agno model từ config.py
    """
    if OpenAIChat is None:
        raise ImportError("Agno not installed. Run: pip install agno")

    return OpenAIChat(
        id=model_name,
        api_key=OPENAI_API_KEY,
        # Agno sẽ handle json_mode nếu cần
    )


async def create_agno_agent(
    model_name: str = DEFAULT_MODEL,
    json_mode: bool = False,
) -> AgentType:  # type: ignore
    """
    Tạo Agno Agent với ConversationManager (có memory)
    
    Args:
        model_name: Model name từ config.py
        json_mode: Enable JSON output
    
    Returns:
        Configured Agno Agent
    """
    # Tạo model từ config
    model = create_agno_model(model_name, json_mode)

    # Lấy tools (đã convert sang Agno format)
    tools = get_agno_tools()

    # Lấy system prompt
    system_prompt = get_system_prompt()

    # Lấy ConversationManager (đã implement BaseDb interface)
    db = await get_conversation_manager()

    if Agent is None:
        raise ImportError("Agno not installed. Run: pip install agno")

    # Type cast: ConversationManager implements BaseDb interface (duck typing)
    # Runtime sẽ hoạt động vì ConversationManager có đủ methods cần thiết
    db_cast = cast(BaseDbType, db)  # type: ignore[assignment]

    # Tạo Agno Agent với DB (có memory)
    agent = Agent(
        name="CANIFA Agent",
        model=model,
        db=db_cast,  # Dùng ConversationManager (implement BaseDb interface)
        tools=tools,
        instructions=system_prompt,  # Agno dùng instructions thay vì system_prompt
        add_history_to_context=True,  # Bật history
        num_history_runs=20,  # Load 20 messages gần nhất
        markdown=True,
    )

    logger.info(f"✅ Agno Agent created with model: {model_name} (WITH MEMORY)")
    return agent


# Singleton instance
_agno_agent_instance: AgentType | None = None  # type: ignore


async def get_agno_agent(
    model_name: str = DEFAULT_MODEL,
    json_mode: bool = False,
) -> AgentType:  # type: ignore
    """
    Get or create Agno Agent singleton (với memory)
    """
    global _agno_agent_instance

    if _agno_agent_instance is None:
        # Tạo agent với ConversationManager (có memory)
        _agno_agent_instance = await create_agno_agent(
            model_name=model_name,
            json_mode=json_mode,
        )

    return _agno_agent_instance


def reset_agno_agent():
    """Reset singleton for testing"""
    global _agno_agent_instance
    _agno_agent_instance = None

