"""
CANIFA Data Retrieval Tool - Tối giản cho Agentic Workflow.
Hỗ trợ Hybrid Search: Semantic (Vector) + Metadata Filter.
"""

import asyncio
import json
import logging
from decimal import Decimal

from langchain_core.tools import tool
from pydantic import BaseModel, Field

from agent.tools.product_search_helpers import build_starrocks_query
from common.starrocks_connection import StarRocksConnection

# from langsmith import traceable

logger = logging.getLogger(__name__)


class DecimalEncoder(json.JSONEncoder):
    """Xử lý kiểu Decimal từ Database khi convert sang JSON."""

    def default(self, obj):
        if isinstance(obj, Decimal):
            return float(obj)
        return super().default(obj)


class SearchItem(BaseModel):
    """Cấu trúc một mục tìm kiếm đơn lẻ trong Multi-Search."""

    query: str = Field(
        ...,
        description="Câu hỏi/mục đích tự do của user (đi chơi, dự tiệc, phỏng vấn,...) - dùng cho Semantic Search",
    )
    keywords: str | None = Field(
        ..., description="Từ khóa sản phẩm cụ thể (áo polo, quần jean,...) - dùng cho LIKE search"
    )
    magento_ref_code: str | None = Field(
        ..., description="Mã sản phẩm hoặc mã màu/SKU (Ví dụ: 8TS24W001 hoặc 8TS24W001-SK010)."
    )
    product_line_vn: str | None = Field(..., description="Dòng sản phẩm (Áo phông, Quần short,...)")
    gender_by_product: str | None = Field(..., description="Giới tính: male, female")
    age_by_product: str | None = Field(..., description="Độ tuổi: adult, kids, baby, others")
    master_color: str | None = Field(..., description="Màu sắc chính (Đen/ Black, Trắng/ White,...)")
    material_group: str | None = Field(
        ...,
        description="Nhóm chất liệu. BẮT BUỘC dùng đúng: 'Yarn - Sợi', 'Knit - Dệt Kim', 'Woven - Dệt Thoi', 'Knit/Woven - Dệt Kim/Dệt Thoi'.",
    )
    season: str | None = Field(..., description="Mùa (Spring Summer, Autumn Winter)")
    style: str | None = Field(..., description="Phong cách (Basic Update, Fashion,...)")
    fitting: str | None = Field(..., description="Form dáng (Regular, Slim, Loose,...)")
    form_neckline: str | None = Field(..., description="Kiểu cổ (Crew Neck, V-neck,...)")
    form_sleeve: str | None = Field(..., description="Kiểu tay (Short Sleeve, Long Sleeve,...)")
    price_min: float | None = Field(..., description="Giá thấp nhất")
    price_max: float | None = Field(..., description="Giá cao nhất")
    action: str = Field(..., description="Hành động: 'search' (tìm kiếm) hoặc 'visual_search' (phân tích ảnh)")


class MultiSearchParams(BaseModel):
    """Tham số cho Parallel Multi-Search."""
    searches: list[SearchItem] = Field(..., description="Danh sách các truy vấn tìm kiếm chạy song song")


@tool(args_schema=MultiSearchParams)
# @traceable(run_type="tool", name="data_retrieval_tool")
async def data_retrieval_tool(searches: list[SearchItem]) -> str:
    """
    Siêu công cụ tìm kiếm sản phẩm CANIFA - Hỗ trợ Parallel Multi-Search (Chạy song song nhiều query).

    💡 ĐIỂM ĐẶC BIỆT: 
    Công cụ này cho phép thực hiện NHIỀU truy vấn tìm kiếm CÙNG LÚC. 
    Hãy dùng nó khi cần SO SÁNH sản phẩm hoặc tìm trọn bộ OUTFIT (mix & match).

    ⚠️ QUAN TRỌNG - KHI NÀO DÙNG GÌ:

    1️⃣ DÙNG 'query' (Semantic Search - BUỘC PHẢI CÓ):
       - Áp dụng cho mọi lượt search để cung cấp bối cảnh (context).
       - Ví dụ: "áo thun nam đi biển", "quần tây công sở", "đồ cho bé màu xanh"...

    2️⃣ DÙNG METADATA FILTERS (Exact/Partial Match):
       - Khi khách nói rõ THUỘC TÍNH: Màu sắc, giá, giới tính, độ tuổi, mã sản phẩm.
       - **QUY TẮC MÃ SẢN PHẨM:** Mọi loại mã (VD: `8TS...` hoặc `8TS...-SK...`) → Điền vào `magento_ref_code`.
       - **QUY TẮC CHẤT LIÊU (material_group):** Chỉ dùng: `Yarn - Sợi`, `Knit - Dệt Kim`, `Woven - Dệt Thoi`, `Knit/Woven - Dệt Kim/Dệt Thoi`.

    📝 VÍ DỤ CHI TIẾT (Single Search):
       - Example 1: searches=[{"query": "áo polo nam giá dưới 400k", "keywords": "áo polo", "gender_by_product": "male", "price_max": 400000}]
       - Example 2: searches=[{"query": "sản phẩm mã 8TS24W001", "magento_ref_code": "8TS24W001"}]

    🚀 VÍ DỤ CẤP CAO (Multi-Search Parallel):
       - Example 3 - So sánh: "So sánh áo thun nam đen và áo sơ mi trắng dưới 500k"
         Tool Call: searches=[
             {"query": "áo thun nam màu đen dưới 500k", "keywords": "áo thun", "master_color": "Đen", "gender_by_product": "male", "price_max": 500000},
             {"query": "áo sơ mi nam trắng dưới 500k", "keywords": "áo sơ mi", "master_color": "Trắng", "gender_by_product": "male", "price_max": 500000}
         ]
       - Example 4 - Phối đồ: "Tìm cho mình một cái quần jean và một cái áo khoác để đi chơi"
         Tool Call: searches=[
             {"query": "quần jean đi chơi năng động", "keywords": "quần jean"},
             {"query": "áo khoác đi chơi năng động", "keywords": "áo khoác"}
         ]
       - Example 5 - Cả gia đình: "Tìm áo phông màu xanh cho bố, mẹ và bé trai"
         Tool Call: searches=[
             {"query": "áo phông nam người lớn màu xanh", "keywords": "áo phông", "master_color": "Xanh", "gender_by_product": "male", "age_by_product": "adult"},
             {"query": "áo phông nữ người lớn màu xanh", "keywords": "áo phông", "master_color": "Xanh", "gender_by_product": "female", "age_by_product": "adult"},
             {"query": "áo phông bé trai màu xanh", "keywords": "áo phông", "master_color": "Xanh", "gender_by_product": "male", "age_by_product": "others"}
         ]
    """
    try:
        db = StarRocksConnection()
        
        # 0. Log input parameters (Đúng ý bro)
        logger.info(f"📥 [Tool Input] data_retrieval_tool received {len(searches)} items:")
        for idx, item in enumerate(searches):
            logger.info(f"   🔹 Item [{idx}]: {item.dict(exclude_none=True)}")
        
        # 1. Tạo tasks chạy song song (Parallel)
        tasks = []
        for item in searches:
            tasks.append(_execute_single_search(db, item))

        logger.info(f"🚀 [Parallel Search] Executing {len(searches)} queries simultaneously...")
        results = await asyncio.gather(*tasks)

        # 2. Tổng hợp kết quả
        combined_results = []
        for i, products in enumerate(results):
            combined_results.append({
                "search_index": i,
                "search_criteria": searches[i].dict(exclude_none=True),
                "count": len(products),
                "products": products
            })

        return json.dumps(
            {"status": "success", "results": combined_results},
            ensure_ascii=False,
            cls=DecimalEncoder
        )

    except Exception as e:
        logger.error(f"Error in Multi-Search data_retrieval_tool: {e}")
        return json.dumps({"status": "error", "message": str(e)})


async def _execute_single_search(db: StarRocksConnection, item: SearchItem) -> list[dict]:
    """Thực thi một search query đơn lẻ (Async)."""
    try:
        # build_starrocks_query handles embedding internally (async)
        sql = await build_starrocks_query(item)
        products = await db.execute_query_async(sql)
        return _format_product_results(products)
    except Exception as e:
        logger.error(f"Single search error for item {item}: {e}")
        return []


def _format_product_results(products: list[dict]) -> list[dict]:
    """Lọc và format kết quả trả về cho Agent."""
    allowed_fields = {
        "internal_ref_code",
        "magento_ref_code",
        "product_color_code",
        "product_name",
        "color_code",
        "master_color",
        "product_color_name",
        "season_sale",
        "season",
        "style",
        "fitting",
        "size_scale",
        "graphic",
        "pattern",
        "weaving",
        "shape_detail",
        "form_neckline",
        "form_sleeve",
        "form_length",
        "form_waistline",
        "form_shoulderline",
        "material",
        "product_group",
        "product_line_vn",
        "unit_of_measure",
        "sale_price",
        "original_price",
        "material_group",
        "product_line_en",
        "age_by_product",
        "gender_by_product",
        "product_image_url",
        "description_text",
        "product_image_url_thumbnail",
        "product_web_url",
        "product_web_material",
    }
    return [{k: v for k, v in p.items() if k in allowed_fields} for p in products[:5]]

