"""
Fashion Q&A Agent Controller
Switched to LangSmith for tracing (configured via environment variables).
"""

import json
import logging
import uuid

from fastapi import BackgroundTasks
from langchain_core.messages import AIMessage, HumanMessage, ToolMessage
from langchain_core.runnables import RunnableConfig

from common.conversation_manager import ConversationManager, get_conversation_manager
from common.llm_factory import create_llm
from config import DEFAULT_MODEL

from .graph import build_graph
from .models import AgentState, get_config
from .tools.get_tools import get_all_tools

logger = logging.getLogger(__name__)


async def chat_controller(
    query: str,
    user_id: str,
    background_tasks: BackgroundTasks,
    model_name: str = DEFAULT_MODEL,
    images: list[str] | None = None,
) -> dict:
    """
    Controller main logic for non-streaming chat requests.
    1. Initialize resources (LLM, tools, graph, conversation manager)
    """
    logger.info(f"▶️ Starting chat_controller with model: {model_name} for user: {user_id}")
    config = get_config()
    config.model_name = model_name

    # Enable JSON mode to ensure structured output
    llm = create_llm(model_name=model_name, streaming=False, json_mode=True)
    tools = get_all_tools()

    graph = build_graph(config, llm=llm, tools=tools)

    # Init ConversationManager (Singleton)
    memory = await get_conversation_manager()

    # LOAD HISTORY & Prepare State
    history_dicts = await memory.get_chat_history(user_id, limit=20)

    # Convert to BaseMessage objects
    history = []
    for h in reversed(history_dicts): 
        if h["is_human"]:
            history.append(HumanMessage(content=h["message"]))
        else:
            history.append(AIMessage(content=h["message"]))

    initial_state, exec_config = _prepare_execution_context(
        query=query, user_id=user_id, history=history, images=images
    )
    try:
        result = await graph.ainvoke(initial_state, config=exec_config)
        # logger.info(f"Answer result from ai: {result}")

        # take ai message from result
        final_ai_message = result.get("ai_response")

        # Extract product IDs from tool messages
        product_ids = _extract_product_ids(result.get("messages", []))

        # Save to DB in background after response is sent
        background_tasks.add_task(
            _handle_post_chat_async,
            memory=memory,
            user_id=user_id,
            human_query=query,
            ai_msg=final_ai_message,
        )

        logger.info(f"✅ Request completed for user {user_id} with {len(product_ids)} products")
        return {
            "ai_response": final_ai_message.content if final_ai_message else "",
            "product_ids": product_ids,
        }

    except Exception as e:
        logger.error(f"💥 Chat error: {e}", exc_info=True)
        raise


def _extract_product_ids(messages: list) -> list[str]:
    """
    Extract product internal_ref_code from tool messages (data_retrieval_tool results).
    Returns list of unique product IDs.
    """
    product_ids = []

    for msg in messages:
        if isinstance(msg, ToolMessage):
            try:
                # Tool result is JSON string
                tool_result = json.loads(msg.content)

                # Check if tool returned products
                if tool_result.get("status") == "success" and "products" in tool_result:
                    for product in tool_result["products"]:
                        product_id = product.get("internal_ref_code")
                        if product_id and product_id not in product_ids:
                            product_ids.append(product_id)
            except (json.JSONDecodeError, KeyError, TypeError) as e:
                logger.debug(f"Could not parse tool message for product IDs: {e}")
                continue

    return product_ids


def _prepare_execution_context(query: str, user_id: str, history: list, images: list | None):
    """Prepare initial state and execution config for the graph run."""
    initial_state: AgentState = {
        "user_query": HumanMessage(content=query),
        "messages": [HumanMessage(content=query)],
        "history": history,
        "user_id": user_id,
        "images_embedding": [],
        "ai_response": None,
    }
    run_id = str(uuid.uuid4())
    # Metadata for LangSmith
    metadata = {"user_id": user_id, "run_id": run_id}

    exec_config = RunnableConfig(
        configurable={
            "user_id": user_id,
            "transient_images": images or [],
            "run_id": run_id,
        },
        run_id=run_id,
        metadata=metadata,  # Attach metadata for LangSmith
    )
    return initial_state, exec_config


async def _handle_post_chat_async(
    memory: ConversationManager, user_id: str, human_query: str, ai_msg: AIMessage | None
):
    """Save chat history in background task after response is sent."""
    if ai_msg:
        try:
            await memory.save_conversation_turn(user_id, human_query, ai_msg.content)
            logger.debug(f"Saved conversation for user {user_id}")
        except Exception as e:
            logger.error(f"Failed to save conversation for user {user_id}: {e}", exc_info=True)
