"""
Test API Routes - Tất cả endpoints cho testing (isolated)
KHÔNG ĐỘNG VÀO chatbot_route.py chính!
"""

import asyncio
import logging
import random

from fastapi import APIRouter, HTTPException
from pydantic import BaseModel, Field

from agent.models import QueryRequest
from common.load_test_manager import get_load_test_manager

router = APIRouter(prefix="/test", tags=["Testing & Load Test"])
logger = logging.getLogger(__name__)


# ==================== MOCK CHAT ENDPOINT ====================

@router.post("/chat-mock", summary="Mock Chat API (for Load Testing)")
async def mock_chat(req: QueryRequest):
    """
    Endpoint MOCK để test performance KHÔNG tốn tiền OpenAI.
    Trả về response giả lập với latency ngẫu nhiên.
    
    ⚠️ CHỈ DÙNG CHO LOAD TESTING!
    """
    # Giả lập latency của real API (100-500ms)
    await asyncio.sleep(random.uniform(0.1, 0.5))
    
    # Mock responses
    mock_responses = [
        "Dạ em đã tìm được một số mẫu áo sơ mi nam đẹp cho anh/chị ạ. Anh/chị có thể xem các sản phẩm sau đây.",
        "Em xin gợi ý một số mẫu áo thun nam phù hợp với yêu cầu của anh/chị.",
        "Dạ, em có tìm thấy một số mẫu quần jean nam trong khoảng giá anh/chị yêu cầu ạ.",
        "Em xin giới thiệu các mẫu áo khoác nam đang có khuyến mãi tốt ạ.",
        "Anh/chị có thể tham khảo các mẫu giày thể thao nam đang được ưa chuộng nhất.",
    ]
    
    # Mock product IDs
    mock_product_ids = [
        f"MOCK_PROD_{random.randint(1000, 9999)}" 
        for _ in range(random.randint(2, 5))
    ]
    
    return {
        "status": "success",
        "ai_response": random.choice(mock_responses),
        "product_ids": mock_product_ids,
        "_mock": True,  # Flag để biết đây là mock response
        "_latency_ms": random.randint(100, 500)
    }


@router.post("/db-search", summary="DB Search Mock (Test StarRocks Performance)")
async def mock_db_search(req: QueryRequest):
    """
    Endpoint để test PERFORMANCE của StarRocks DB query.
    Hỗ trợ Multi-Search (Parallel).
    """
    from agent.tools.data_retrieval_tool import data_retrieval_tool
    
    try:
        # Mock Multi-Search call (Parallel)
        tool_result = await data_retrieval_tool.ainvoke({
            "searches": [
                {
                    "keywords": "áo sơ mi",
                    "gender_by_product": "male",
                    "price_max": 500000
                },
                {
                    "keywords": "quần jean",
                    "gender_by_product": "male",
                    "price_max": 800000
                }
            ]
        })
        
        # Parse result
        import json
        result_data = json.loads(tool_result)
        
        # Collect all product IDs from all search results
        all_product_ids = []
        if result_data.get("status") == "success":
            for res in result_data.get("results", []):
                ids = [p.get("internal_ref_code", "") for p in res.get("products", [])]
                all_product_ids.extend(ids)
        
        return {
            "status": "success",
            "ai_response": "Kết quả Multi-Search Parallel từ DB",
            "product_ids": list(set(all_product_ids)),
            "_db_test": True,
            "_queries_count": len(result_data.get("results", [])),
            "_total_products": len(all_product_ids)
        }
        
    except Exception as e:
        logger.error(f"DB multi-search error: {e}")
        return {
            "status": "error",
            "ai_response": f"Lỗi: {str(e)}",
            "product_ids": [],
            "_error": str(e)
        }


# ==================== LOAD TEST CONTROL ====================

class StartTestRequest(BaseModel):
    """Request body để start test"""
    target_url: str = Field(default="http://localhost:5000", description="Base URL của target")
    num_users: int = Field(default=10, ge=1, le=1000, description="Số lượng concurrent users")
    spawn_rate: int = Field(default=2, ge=1, le=100, description="Tốc độ spawn users (users/second)")
    duration_seconds: int = Field(default=60, ge=10, le=600, description="Thời gian chạy test (giây)")
    test_type: str = Field(default="chat_mock", description="chat_mock | chat_real | history")


@router.post("/loadtest/start", summary="Bắt đầu Load Test")
async def start_load_test(req: StartTestRequest):
    """
    Bắt đầu load test với config được chỉ định.
    
    **test_type options:**
    - `chat_mock`: Test mock chat API (KHÔNG tốn tiền) ⭐ Khuyên dùng
    - `chat_real`: Test real chat API (TỐN TIỀN OpenAI!)
    - `history`: Test history API (không tốn tiền LLM)
    """
    try:
        manager = get_load_test_manager()
        
        config_dict = req.model_dump()
        result = manager.start_test(config_dict)
        
        if "error" in result:
            raise HTTPException(status_code=400, detail=result["error"])
        
        return {
            "status": "success",
            "message": "Load test started",
            "data": result
        }
    
    except Exception as e:
        logger.error(f"Error starting load test: {e}")
        raise HTTPException(status_code=500, detail=str(e))


@router.post("/loadtest/stop", summary="Dừng Load Test")
async def stop_load_test():
    """Dừng load test đang chạy"""
    try:
        manager = get_load_test_manager()
        result = manager.stop_test()
        
        if "error" in result:
            raise HTTPException(status_code=400, detail=result["error"])
        
        return {
            "status": "success",
            "message": "Load test stopped",
            "data": result
        }
    
    except Exception as e:
        logger.error(f"Error stopping load test: {e}")
        raise HTTPException(status_code=500, detail=str(e))


@router.get("/loadtest/metrics", summary="Lấy Metrics Realtime")
async def get_load_test_metrics():
    """
    Lấy metrics realtime của load test.
    Frontend poll endpoint này mỗi 2 giây.
    """
    try:
        manager = get_load_test_manager()
        metrics = manager.get_metrics()
        
        return {
            "status": "success",
            "data": metrics
        }
    
    except Exception as e:
        logger.error(f"Error getting metrics: {e}")
        raise HTTPException(status_code=500, detail=str(e))


@router.get("/loadtest/status", summary="Check Test Status")
async def get_load_test_status():
    """Check xem load test có đang chạy không"""
    try:
        manager = get_load_test_manager()
        
        return {
            "status": "success",
            "data": {
                "is_running": manager.is_running(),
                "current_status": manager.status
            }
        }
    
    except Exception as e:
        logger.error(f"Error getting status: {e}")
        raise HTTPException(status_code=500, detail=str(e))
