"""
🚀 QUICK BENCHMARK TOOL - Test nhanh performance
============================================================
Multiple testing methods trong 1 file:
1. Pure Python asyncio benchmark
2. Apache Bench wrapper
3. wrk wrapper (if installed)
4. Quick Locust headless test

USAGE:
------
python locust/quick_benchmark.py --method all
python locust/quick_benchmark.py --method ab
python locust/quick_benchmark.py --method locust
python locust/quick_benchmark.py --endpoint /api/agent/chat
"""

import argparse
import asyncio
import json
import subprocess
import sys
import time
from pathlib import Path
from typing import List, Dict

import aiohttp

# ============================================================
# 1. PURE PYTHON ASYNC BENCHMARK
# ============================================================
async def async_benchmark(
    url: str,
    total_requests: int = 100,
    concurrent: int = 10,
    payload: Dict = None
):
    """
    Pure Python async benchmark - không cần tool ngoài.
    Giống ab nhưng flexible hơn.
    """
    print(f"\n🐍 PYTHON ASYNC BENCHMARK")
    print("="*60)
    print(f"URL: {url}")
    print(f"Total Requests: {total_requests}")
    print(f"Concurrency: {concurrent}")
    print("="*60)
    
    results = {
        "success": 0,
        "failed": 0,
        "response_times": []
    }
    
    async def single_request(session, sem):
        async with sem:
            start = time.time()
            try:
                if payload:
                    async with session.post(url, json=payload) as resp:
                        await resp.text()
                        duration = (time.time() - start) * 1000
                        results["response_times"].append(duration)
                        if resp.status == 200:
                            results["success"] += 1
                        else:
                            results["failed"] += 1
                else:
                    async with session.get(url) as resp:
                        await resp.text()
                        duration = (time.time() - start) * 1000
                        results["response_times"].append(duration)
                        if resp.status == 200:
                            results["success"] += 1
                        else:
                            results["failed"] += 1
            except Exception as e:
                results["failed"] += 1
                print(f"❌ Error: {e}")
    
    # Run benchmark
    sem = asyncio.Semaphore(concurrent)
    start_time = time.time()
    
    async with aiohttp.ClientSession() as session:
        tasks = [single_request(session, sem) for _ in range(total_requests)]
        await asyncio.gather(*tasks)
    
    total_time = time.time() - start_time
    
    # Calculate stats
    if results["response_times"]:
        sorted_times = sorted(results["response_times"])
        p50 = sorted_times[len(sorted_times) // 2]
        p95 = sorted_times[int(len(sorted_times) * 0.95)]
        p99 = sorted_times[int(len(sorted_times) * 0.99)]
        avg = sum(sorted_times) / len(sorted_times)
        
        print(f"\n📊 RESULTS:")
        print(f"Total Time: {total_time:.2f}s")
        print(f"Requests/sec: {total_requests / total_time:.2f}")
        print(f"Success: {results['success']}")
        print(f"Failed: {results['failed']}")
        print(f"\nResponse Times:")
        print(f"  Avg: {avg:.2f}ms")
        print(f"  P50: {p50:.2f}ms")
        print(f"  P95: {p95:.2f}ms")
        print(f"  P99: {p99:.2f}ms")
        print(f"  Min: {min(sorted_times):.2f}ms")
        print(f"  Max: {max(sorted_times):.2f}ms")
    
    return results


# ============================================================
# 2. APACHE BENCH WRAPPER
# ============================================================
def run_apache_bench(url: str, requests: int = 1000, concurrency: int = 10):
    """
    Sử dụng Apache Bench (ab) nếu có cài.
    """
    print(f"\n🔨 APACHE BENCH TEST")
    print("="*60)
    
    try:
        cmd = [
            "ab",
            "-n", str(requests),
            "-c", str(concurrency),
            "-T", "application/json",
            url
        ]
        
        result = subprocess.run(cmd, capture_output=True, text=True)
        print(result.stdout)
        
        if result.returncode != 0:
            print(f"❌ Error: {result.stderr}")
            
    except FileNotFoundError:
        print("❌ Apache Bench (ab) not found!")
        print("💡 Install: apt-get install apache2-utils (Linux)")
        print("           or brew install ab (Mac)")
        return None


# ============================================================
# 3. WRK WRAPPER
# ============================================================
def run_wrk(url: str, duration: str = "10s", threads: int = 4, connections: int = 10):
    """
    Sử dụng wrk nếu có cài - tool siêu nhanh viết bằng C.
    """
    print(f"\n⚡ WRK BENCHMARK")
    print("="*60)
    
    try:
        cmd = [
            "wrk",
            "-t", str(threads),
            "-c", str(connections),
            "-d", duration,
            url
        ]
        
        result = subprocess.run(cmd, capture_output=True, text=True)
        print(result.stdout)
        
        if result.returncode != 0:
            print(f"❌ Error: {result.stderr}")
            
    except FileNotFoundError:
        print("❌ wrk not found!")
        print("💡 Install: https://github.com/wg/wrk")
        print("   Mac: brew install wrk")
        print("   Linux: build from source")
        return None


# ============================================================
# 4. LOCUST HEADLESS
# ============================================================
def run_locust_headless(
    locustfile: str,
    host: str,
    users: int = 50,
    spawn_rate: int = 10,
    run_time: str = "30s"
):
    """
    Chạy Locust ở chế độ headless (không UI).
    """
    print(f"\n🦗 LOCUST HEADLESS TEST")
    print("="*60)
    
    try:
        cmd = [
            "locust",
            "-f", locustfile,
            "--headless",
            "--host", host,
            "-u", str(users),
            "-r", str(spawn_rate),
            "--run-time", run_time,
            "--html", "locust_report.html"
        ]
        
        print(f"Running: {' '.join(cmd)}")
        result = subprocess.run(cmd, text=True)
        
        if result.returncode == 0:
            print("\n✅ Test completed!")
            print(f"📊 Report saved to: locust_report.html")
        
    except FileNotFoundError:
        print("❌ Locust not found!")
        print("💡 Install: pip install locust")
        return None


# ============================================================
# 5. K6 WRAPPER (Bonus)
# ============================================================
def generate_k6_script(url: str, output_file: str = "test.js"):
    """
    Generate k6 test script.
    """
    script = f"""
import http from 'k6/http';
import {{ check, sleep }} from 'k6';

export let options = {{
  stages: [
    {{ duration: '30s', target: 20 }},
    {{ duration: '1m', target: 50 }},
    {{ duration: '30s', target: 0 }},
  ],
  thresholds: {{
    http_req_duration: ['p(95)<500'], // 95% requests < 500ms
  }},
}};

export default function () {{
  let res = http.post('{url}', JSON.stringify({{
    message: 'test',
    user_id: 'test_user',
    thread_id: 'test_thread'
  }}), {{
    headers: {{ 'Content-Type': 'application/json' }},
  }});
  
  check(res, {{
    'status is 200': (r) => r.status === 200,
    'response time < 2000ms': (r) => r.timings.duration < 2000,
  }});
  
  sleep(1);
}}
"""
    
    with open(output_file, "w") as f:
        f.write(script)
    
    print(f"✅ k6 script generated: {output_file}")
    print(f"💡 Run with: k6 run {output_file}")


# ============================================================
# MAIN CLI
# ============================================================
def main():
    parser = argparse.ArgumentParser(description="Quick Performance Benchmark Tool")
    parser.add_argument("--method", default="python", 
                       choices=["python", "ab", "wrk", "locust", "k6-gen", "all"],
                       help="Benchmark method to use")
    parser.add_argument("--endpoint", default="/api/agent/chat",
                       help="API endpoint to test")
    parser.add_argument("--host", default="http://localhost:8000",
                       help="Server host")
    parser.add_argument("--requests", type=int, default=100,
                       help="Total requests (for python/ab)")
    parser.add_argument("--concurrency", type=int, default=10,
                       help="Concurrent requests")
    
    args = parser.parse_args()
    
    url = args.host + args.endpoint
    
    print("\n" + "="*60)
    print("🔥 QUICK BENCHMARK TOOL")
    print("="*60)
    print(f"Target: {url}")
    print("="*60)
    
    # Prepare payload for chat endpoint
    payload = None
    if "/chat" in args.endpoint:
        payload = {
            "message": "test performance",
            "user_id": "benchmark_user",
            "thread_id": "benchmark_thread"
        }
    
    if args.method == "python" or args.method == "all":
        asyncio.run(async_benchmark(url, args.requests, args.concurrency, payload))
    
    if args.method == "ab" or args.method == "all":
        # ab không support POST JSON dễ, skip hoặc dùng cho GET
        if not payload:
            run_apache_bench(url, args.requests, args.concurrency)
        else:
            print("\n⚠️  Apache Bench: Skipped (POST JSON not supported easily)")
    
    if args.method == "wrk" or args.method == "all":
        run_wrk(url, "30s", 4, args.concurrency)
    
    if args.method == "locust":
        locustfile = str(Path(__file__).parent / "locustfile_production.py")
        run_locust_headless(locustfile, args.host, 50, 10, "30s")
    
    if args.method == "k6-gen":
        generate_k6_script(url)
    
    print("\n" + "="*60)
    print("✅ Benchmark Complete!")
    print("="*60 + "\n")


if __name__ == "__main__":
    main()
