"""
🎯 Production-like Performance Testing with Locust
Test thực tế các endpoint QUAN TRỌNG của chatbot để tìm bottleneck.

CÁCH CHẠY:
1. Start API server:
   cd backend
   python run.py

2. Run Locust với file này:
   locust -f locust/locustfile_production.py --host=http://localhost:8000

3. Mở browser: http://localhost:8089
   - Number of users: 50 (tăng dần để test scalability)
   - Spawn rate: 10 users/second
   - Host: http://localhost:8000

📊 METRICS CẦN QUAN TÂM:
- Response Time (P50, P95, P99): Thời gian phản hồi
- RPS (Requests per second): Throughput
- Failure Rate: Tỷ lệ lỗi
"""

import random
import time

from locust import HttpUser, between, events, task

# ============================================================
# TEST DATA - Mô phỏng real user queries
# ============================================================
CHAT_QUERIES = [
    # Simple product search (nhanh)
    "tìm áo phông nam",
    "quần jean nữ",
    "váy liền",
    # Complex search với filters (chậm hơn)
    "áo khoác nam màu đen giá dưới 500k",
    "quần short nữ mùa hè",
    # Knowledge queries (test RAG knowledge base)
    "Canifa có cửa hàng ở Hà Nội không?",
    "chính sách đổi trả",
    "bảng size áo nam",
    # Conversational (test LLM reasoning)
    "xin chào",
    "cảm ơn bạn",
]

VISUAL_SEARCH_IMAGES = [
    "https://canifa.com/img/500/750/resize/6/t/6ts24w007-sk010-m-1-u.webp",
    "https://canifa.com/img/500/750/resize/6/a/6aw24w006-sb492-thumb-2.webp",
]


# ============================================================
# CUSTOM METRICS - Track specific bottlenecks
# ============================================================
chat_response_times = []
db_query_times = []
llm_call_times = []


@events.test_stop.add_listener
def on_test_stop(environment, **kwargs):
    """Print summary khi test xong."""
    print("\n" + "=" * 60)
    print("📊 PERFORMANCE TEST SUMMARY")
    print("=" * 60)

    if chat_response_times:
        avg_chat = sum(chat_response_times) / len(chat_response_times)
        print(f"💬 Chat Response Time (avg): {avg_chat:.2f}ms")
        print(f"   P50: {sorted(chat_response_times)[len(chat_response_times) // 2]:.2f}ms")
        print(f"   P95: {sorted(chat_response_times)[int(len(chat_response_times) * 0.95)]:.2f}ms")

    print("\n🎯 RECOMMENDED ACTIONS:")
    print("1. Check response times > 2000ms")
    print("2. Identify which endpoint has highest latency")
    print("3. Use profiling tools (below) for deep dive")
    print("=" * 60 + "\n")


# ============================================================
# USER BEHAVIORS
# ============================================================


class ChatbotUser(HttpUser):
    """Mô phỏng user sử dụng chatbot."""

    wait_time = between(2, 5)  # User thường đợi 2-5s giữa các câu hỏi

    def on_start(self):
        """Initialize user - giống như user mở chat lần đầu."""
        self.user_id = f"user_{random.randint(1000, 9999)}"
        self.thread_id = f"thread_{int(time.time() * 1000)}_{random.randint(0, 999)}"

    @task(10)
    def chat_message(self):
        """
        Task quan trọng nhất - test FULL chatbot flow.
        Weight: 10 (chiếm 10/12 = 83% traffic)
        """
        query = random.choice(CHAT_QUERIES)

        start_time = time.time()
        with self.client.post(
            "/api/agent/chat",
            json={"message": query, "user_id": self.user_id, "thread_id": self.thread_id},
            catch_response=True,
            name="/api/agent/chat [MAIN]",
        ) as response:
            duration = (time.time() - start_time) * 1000  # ms
            chat_response_times.append(duration)

            if response.status_code == 200:
                response.success()
            else:
                response.failure(f"Status: {response.status_code}")

    @task(1)
    def health_check(self):
        """
        Health check endpoint.
        Weight: 1 (8% traffic)
        """
        self.client.get("/", name="/health")

    @task(1)
    def visual_search(self):
        """
        Test visual search nếu có.
        Weight: 1 (8% traffic)
        """
        img_url = random.choice(VISUAL_SEARCH_IMAGES)
        self.client.post(
            "/api/recommend/image",
            json={"image_url": img_url},
            name="/api/recommend/image [VISUAL]",
            catch_response=True,
        )


class DatabaseStressUser(HttpUser):
    """
    Riêng để test ONLY database bottleneck.
    Chạy riêng test này để isolate DB performance.
    """

    wait_time = between(8, 20)  # Mô phỏng người dùng thật: đợi 8-20s giữa các lần search

    @task
    def db_direct_query(self):
        """Test trực tiếp StarRocks query speed."""
        self.client.post(
            "/test/db-search",
            json={"query": random.choice(["áo phông", "quần jean", "váy"]), "limit": 10},
            name="/test/db-search [DB_ONLY]",
        )


# ============================================================
# SHAPE CONFIGURATION - Simulate real traffic pattern
# ============================================================
class WebsiteUserShape:
    """
    Mô phỏng traffic pattern thực tế:
    - Sáng sớm: ít user
    - Trưa + tối: peak hours
    - Đêm khuya: vắng

    CHỈ DÙNG KHI MUỐN TEST THEO PATTERN, KHÔNG DÙNG CHO TEST NHANH.
    """

    pass  # Implement nếu cần


if __name__ == "__main__":
    print("🚀 Starting Locust Performance Test...")
    print("📖 Read the docstring for instructions!")
