#!/usr/bin/env python3
"""
🦗 LOCUST MCP SERVER - Custom implementation
============================================================
MCP Server wrapper cho Locust performance testing.
Tích hợp trực tiếp vào project chatbot Canifa.

USAGE:
------
Add vào MCP config (mcp_config.json):
{
  "mcpServers": {
    "locust-canifa": {
      "command": "python",
      "args": [
        "d:\\cnf\\chatbot_canifa\\backend\\locust\\locust_mcp_server.py"
      ],
      "env": {
        "LOCUST_HOST": "http://localhost:8000",
        "LOCUST_USERS": "50",
        "LOCUST_SPAWN_RATE": "10",
        "LOCUST_RUN_TIME": "30s"
      }
    }
  }
}

Sau đó hỏi AI: "Run locust test for locustfile_production.py"
"""

import asyncio
import json
import logging
import os
import subprocess
import sys
from pathlib import Path
from typing import Any, Optional

# MCP SDK imports
try:
    from mcp.server import Server
    from mcp.server.stdio import stdio_server
    from mcp.types import Tool, TextContent
except ImportError:
    print("ERROR: MCP SDK not installed!", file=sys.stderr)
    print("Install with: pip install mcp", file=sys.stderr)
    sys.exit(1)

logging.basicConfig(level=logging.INFO)
logger = logging.getLogger("locust-mcp")

# ============================================================
# DEFAULT CONFIGS
# ============================================================
DEFAULT_CONFIG = {
    "host": os.getenv("LOCUST_HOST", "http://localhost:8000"),
    "users": int(os.getenv("LOCUST_USERS", "50")),
    "spawn_rate": int(os.getenv("LOCUST_SPAWN_RATE", "10")),
    "run_time": os.getenv("LOCUST_RUN_TIME", "30s"),
    "locust_dir": str(Path(__file__).parent),
}

# ============================================================
# MCP SERVER IMPLEMENTATION
# ============================================================
app = Server("locust-canifa")


@app.list_tools()
async def list_tools() -> list[Tool]:
    """List available Locust testing tools."""
    return [
        Tool(
            name="run_locust_test",
            description="""
            Run Locust performance test for Canifa chatbot.
            
            Parameters:
            - test_file: Locust test file (e.g., 'locustfile_production.py')
            - headless: Run without UI (default: True)
            - host: Target server (default: http://localhost:8000)
            - users: Number of concurrent users (default: 50)
            - spawn_rate: Users spawned per second (default: 10)
            - run_time: Test duration (e.g., '30s', '2m', default: 30s)
            
            Returns: Test results including RPS, response times, and failure rate.
            """,
            inputSchema={
                "type": "object",
                "properties": {
                    "test_file": {
                        "type": "string",
                        "description": "Locust test file name (e.g., 'locustfile_production.py')",
                        "default": "locustfile_production.py"
                    },
                    "headless": {
                        "type": "boolean",
                        "description": "Run in headless mode (no UI)",
                        "default": True
                    },
                    "host": {
                        "type": "string",
                        "description": "Target host URL",
                        "default": DEFAULT_CONFIG["host"]
                    },
                    "users": {
                        "type": "integer",
                        "description": "Number of concurrent users",
                        "default": DEFAULT_CONFIG["users"]
                    },
                    "spawn_rate": {
                        "type": "integer",
                        "description": "User spawn rate (users/second)",
                        "default": DEFAULT_CONFIG["spawn_rate"]
                    },
                    "run_time": {
                        "type": "string",
                        "description": "Test duration (e.g., '30s', '2m')",
                        "default": DEFAULT_CONFIG["run_time"]
                    }
                },
                "required": ["test_file"]
            }
        ),
        Tool(
            name="list_locust_files",
            description="List all available Locust test files in the project",
            inputSchema={
                "type": "object",
                "properties": {}
            }
        ),
        Tool(
            name="quick_performance_check",
            description="""
            Quick performance check using Python async benchmark (no Locust install needed).
            Fast way to check if chatbot is responsive.
            
            Parameters:
            - endpoint: API endpoint to test (default: /api/agent/chat)
            - requests: Total requests (default: 50)
            - concurrency: Concurrent requests (default: 10)
            """,
            inputSchema={
                "type": "object",
                "properties": {
                    "endpoint": {
                        "type": "string",
                        "default": "/api/agent/chat"
                    },
                    "requests": {
                        "type": "integer",
                        "default": 50
                    },
                    "concurrency": {
                        "type": "integer",
                        "default": 10
                    }
                }
            }
        )
    ]


@app.call_tool()
async def call_tool(name: str, arguments: Any) -> list[TextContent]:
    """Execute Locust testing tools."""
    
    if name == "run_locust_test":
        return await run_locust_test(arguments)
    
    elif name == "list_locust_files":
        return await list_locust_files()
    
    elif name == "quick_performance_check":
        return await quick_performance_check(arguments)
    
    else:
        raise ValueError(f"Unknown tool: {name}")


# ============================================================
# TOOL IMPLEMENTATIONS
# ============================================================

async def run_locust_test(args: dict) -> list[TextContent]:
    """Run Locust performance test."""
    
    test_file = args.get("test_file", "locustfile_production.py")
    headless = args.get("headless", True)
    host = args.get("host", DEFAULT_CONFIG["host"])
    users = args.get("users", DEFAULT_CONFIG["users"])
    spawn_rate = args.get("spawn_rate", DEFAULT_CONFIG["spawn_rate"])
    run_time = args.get("run_time", DEFAULT_CONFIG["run_time"])
    
    locust_file_path = Path(DEFAULT_CONFIG["locust_dir"]) / test_file
    
    if not locust_file_path.exists():
        return [TextContent(
            type="text",
            text=f"❌ Error: Locust file not found: {locust_file_path}\n\n"
                 f"Available files:\n{await _get_locust_files_text()}"
        )]
    
    # Build Locust command
    cmd = [
        "locust",
        "-f", str(locust_file_path),
        "--host", host,
        "-u", str(users),
        "-r", str(spawn_rate),
        "--run-time", run_time,
    ]
    
    if headless:
        cmd.append("--headless")
        # Add stats output
        cmd.extend(["--html", "locust_report.html"])
        cmd.extend(["--csv", "locust_stats"])
    
    logger.info(f"Running Locust: {' '.join(cmd)}")
    
    try:
        # Run Locust
        result = subprocess.run(
            cmd,
            capture_output=True,
            text=True,
            timeout=300  # 5 minutes max
        )
        
        output = result.stdout + result.stderr
        
        # Parse results
        summary = _parse_locust_output(output)
        
        response = f"""
🦗 LOCUST PERFORMANCE TEST RESULTS
{'='*60}

📝 Test Configuration:
  - File: {test_file}
  - Host: {host}
  - Users: {users}
  - Spawn Rate: {spawn_rate}/s
  - Duration: {run_time}

{summary}

📊 Detailed Output:
{output}

💾 Reports Generated:
  - HTML Report: locust_report.html
  - CSV Stats: locust_stats_*.csv
"""
        
        return [TextContent(type="text", text=response)]
    
    except subprocess.TimeoutExpired:
        return [TextContent(
            type="text",
            text="❌ Test timeout! Locust ran for more than 5 minutes."
        )]
    except FileNotFoundError:
        return [TextContent(
            type="text",
            text="❌ Locust not installed!\n\n"
                 "Install with: pip install locust"
        )]
    except Exception as e:
        logger.error(f"Locust error: {e}")
        return [TextContent(
            type="text",
            text=f"❌ Error running Locust: {str(e)}"
        )]


async def list_locust_files() -> list[TextContent]:
    """List available Locust test files."""
    text = await _get_locust_files_text()
    return [TextContent(type="text", text=text)]


async def _get_locust_files_text() -> str:
    """Get formatted list of Locust files."""
    locust_dir = Path(DEFAULT_CONFIG["locust_dir"])
    locust_files = list(locust_dir.glob("locustfile*.py"))
    
    if not locust_files:
        return "❌ No Locust test files found in locust directory!"
    
    output = "📋 Available Locust Test Files:\n" + "="*60 + "\n\n"
    for f in locust_files:
        output += f"  - {f.name}\n"
    
    return output


async def quick_performance_check(args: dict) -> list[TextContent]:
    """Quick performance check without Locust."""
    
    endpoint = args.get("endpoint", "/api/agent/chat")
    requests = args.get("requests", 50)
    concurrency = args.get("concurrency", 10)
    
    # Run quick benchmark script
    benchmark_script = Path(DEFAULT_CONFIG["locust_dir"]) / "quick_benchmark.py"
    
    if not benchmark_script.exists():
        return [TextContent(
            type="text",
            text="❌ Quick benchmark script not found!"
        )]
    
    cmd = [
        "python",
        str(benchmark_script),
        "--method", "python",
        "--endpoint", endpoint,
        "--requests", str(requests),
        "--concurrency", str(concurrency)
    ]
    
    try:
        result = subprocess.run(cmd, capture_output=True, text=True, timeout=120)
        output = result.stdout + result.stderr
        
        return [TextContent(type="text", text=output)]
    
    except Exception as e:
        return [TextContent(
            type="text",
            text=f"❌ Error running quick benchmark: {str(e)}"
        )]


def _parse_locust_output(output: str) -> str:
    """Parse Locust output to extract key metrics."""
    
    lines = output.split('\n')
    summary = "📊 Key Metrics:\n"
    
    # Look for key metrics in output
    for line in lines:
        if "requests/s" in line.lower() or "rps" in line.lower():
            summary += f"  {line.strip()}\n"
        elif "response time" in line.lower():
            summary += f"  {line.strip()}\n"
        elif "failure" in line.lower():
            summary += f"  {line.strip()}\n"
    
    if summary == "📊 Key Metrics:\n":
        summary += "  (Check detailed output below)\n"
    
    return summary


# ============================================================
# MAIN - Start MCP Server
# ============================================================
async def main():
    """Start the MCP server."""
    logger.info("🦗 Starting Locust MCP Server for Canifa Chatbot...")
    logger.info(f"📁 Locust directory: {DEFAULT_CONFIG['locust_dir']}")
    logger.info(f"🎯 Default host: {DEFAULT_CONFIG['host']}")
    
    async with stdio_server() as (read_stream, write_stream):
        await app.run(
            read_stream,
            write_stream,
            app.create_initialization_options()
        )


if __name__ == "__main__":
    asyncio.run(main())
