"""
Fashion Q&A Agent Controller
Switched to LangSmith for tracing (configured via environment variables).
"""

import json
import logging
import uuid

from fastapi import BackgroundTasks
from langchain_core.messages import AIMessage, HumanMessage, ToolMessage
from langchain_core.runnables import RunnableConfig

from common.conversation_manager import ConversationManager, get_conversation_manager
from common.llm_factory import create_llm
from config import DEFAULT_MODEL

from .graph import build_graph
from .models import AgentState, get_config
from .tools.get_tools import get_all_tools

logger = logging.getLogger(__name__)


async def chat_controller(
    query: str,
    user_id: str,
    background_tasks: BackgroundTasks,
    model_name: str = DEFAULT_MODEL,
    images: list[str] | None = None,
) -> dict:
    """
    Controller main logic for non-streaming chat requests.
    1. Initialize resources (LLM, tools, graph, conversation manager)
    """
    logger.info(f"▶️ Starting chat_controller with model: {model_name} for user: {user_id}")
    config = get_config()
    config.model_name = model_name

    # Enable JSON mode to ensure structured output
    llm = create_llm(model_name=model_name, streaming=False, json_mode=True)
    tools = get_all_tools()

    graph = build_graph(config, llm=llm, tools=tools)

    # Init ConversationManager (Singleton)
    memory = await get_conversation_manager()

    # LOAD HISTORY & Prepare State (Optimize: history logic remains solid)
    history_dicts = await memory.get_chat_history(user_id, limit=20)

    history = []
    for h in reversed(history_dicts):
        msg_cls = HumanMessage if h["is_human"] else AIMessage
        history.append(msg_cls(content=h["message"]))

    initial_state, exec_config = _prepare_execution_context(
        query=query, user_id=user_id, history=history, images=images
    )

    try:
        # TỐI ƯU: Chạy Graph
        result = await graph.ainvoke(initial_state, config=exec_config)

        # TỐI ƯU: Extract IDs từ Tool Messages một lần duy nhất
        all_product_ids = _extract_product_ids(result.get("messages", []))

        # TỐI ƯU: Xử lý AI Response
        ai_raw_content = result.get("ai_response").content if result.get("ai_response") else ""
        logger.info(f"💾 [RAW AI OUTPUT]:\n{ai_raw_content}")

        # Parse JSON để lấy text response và product_ids từ AI
        ai_text_response = ai_raw_content
        try:
            # Vì json_mode=True, OpenAI sẽ nhả raw JSON
            ai_json = json.loads(ai_raw_content)

            # Extract text response từ JSON
            ai_text_response = ai_json.get("ai_response", ai_raw_content)

            # Merge product_ids từ AI JSON (nếu có) - KHÔNG dùng set() vì dict unhashable
            explicit_ids = ai_json.get("product_ids", [])
            if explicit_ids and isinstance(explicit_ids, list):
                # Merge và deduplicate by SKU
                seen_skus = {p["sku"] for p in all_product_ids if "sku" in p}
                for product in explicit_ids:
                    if isinstance(product, dict) and product.get("sku") not in seen_skus:
                        all_product_ids.append(product)
                        seen_skus.add(product.get("sku"))
        except (json.JSONDecodeError, Exception) as e:
            # Nếu AI trả về text thường (hiếm khi xảy ra trong JSON mode) thì ignore
            logger.warning(f"Could not parse AI response as JSON: {e}")
            pass

        # BACKGROUND TASK: Lưu history nhanh gọn
        background_tasks.add_task(
            _handle_post_chat_async,
            memory=memory,
            user_id=user_id,
            human_query=query,
            ai_msg=AIMessage(content=ai_text_response),
        )

        return {
            "ai_response": ai_text_response,  # CHỈ text, không phải JSON
            "product_ids": all_product_ids,  # Array of product objects
        }

    except Exception as e:
        logger.error(f"💥 Chat error for user {user_id}: {e}", exc_info=True)
        raise


def _extract_product_ids(messages: list) -> list[dict]:
    """
    Extract full product info from tool messages (data_retrieval_tool results).
    Returns list of product objects with: sku, name, price, sale_price, url, thumbnail_image_url.
    """
    products = []
    seen_skus = set()

    for msg in messages:
        if isinstance(msg, ToolMessage):
            try:
                # Tool result is JSON string
                tool_result = json.loads(msg.content)

                # Check if tool returned products
                if tool_result.get("status") == "success" and "products" in tool_result:
                    for product in tool_result["products"]:
                        sku = product.get("internal_ref_code")
                        if sku and sku not in seen_skus:
                            seen_skus.add(sku)

                            # Extract full product info
                            product_obj = {
                                "sku": sku,
                                "name": product.get("magento_product_name", ""),
                                "price": product.get("price_vnd", 0),
                                "sale_price": product.get("sale_price_vnd"),  # null nếu không sale
                                "url": product.get("magento_url_key", ""),
                                "thumbnail_image_url": product.get("thumbnail_image_url", ""),
                            }
                            products.append(product_obj)
            except (json.JSONDecodeError, KeyError, TypeError) as e:
                logger.debug(f"Could not parse tool message for products: {e}")
                continue

    return products


def _prepare_execution_context(query: str, user_id: str, history: list, images: list | None):
    """Prepare initial state and execution config for the graph run."""
    initial_state: AgentState = {
        "user_query": HumanMessage(content=query),
        "messages": [HumanMessage(content=query)],
        "history": history,
        "user_id": user_id,
        "images_embedding": [],
        "ai_response": None,
    }
    run_id = str(uuid.uuid4())
    # Metadata for LangSmith (TẮT TẠM VÌ RATE LIMIT)
    # metadata = {"user_id": user_id, "run_id": run_id}

    exec_config = RunnableConfig(
        configurable={
            "user_id": user_id,
            "transient_images": images or [],
            "run_id": run_id,
        },
        run_id=run_id,
        # metadata=metadata,  # Attach metadata for LangSmith
    )
    return initial_state, exec_config


async def _handle_post_chat_async(
    memory: ConversationManager, user_id: str, human_query: str, ai_msg: AIMessage | None
):
    """Save chat history in background task after response is sent."""
    if ai_msg:
        try:
            await memory.save_conversation_turn(user_id, human_query, ai_msg.content)
            logger.debug(f"Saved conversation for user {user_id}")
        except Exception as e:
            logger.error(f"Failed to save conversation for user {user_id}: {e}", exc_info=True)
