"""
Fashion Q&A Agent Router
FastAPI endpoints cho Fashion Q&A Agent service.
Router chỉ chứa định nghĩa API, logic nằm ở controller.
"""

import logging

from fastapi import APIRouter, BackgroundTasks, HTTPException
from opentelemetry import trace

from agent.controller import chat_controller
from agent.models import QueryRequest
from config import DEFAULT_MODEL

logger = logging.getLogger(__name__)
tracer = trace.get_tracer(__name__)
router = APIRouter()


@router.post("/api/agent/chat", summary="Fashion Q&A Chat (Non-streaming)")
async def fashion_qa_chat(req: QueryRequest, background_tasks: BackgroundTasks):
    """
    Endpoint chat không stream - trả về response JSON đầy đủ một lần.
    """
    user_id = req.user_id or "default_user"

    logger.info(f"📥 [Incoming Query - NonStream] User: {user_id} | Query: {req.user_query}")

    # Get current span để add logs VÀO JAEGER UI
    span = trace.get_current_span()
    span.set_attribute("user.id", user_id)
    span.set_attribute("chat.user_query", req.user_query)
    span.add_event(
        "📥 User query received", attributes={"user_id": user_id, "query": req.user_query, "timestamp": "incoming"}
    )

    try:
        # Gọi controller để xử lý logic (Non-streaming)
        result = await chat_controller(
            query=req.user_query,
            user_id=user_id,
            background_tasks=background_tasks,
            model_name=DEFAULT_MODEL,
            images=req.images,
        )

        # Log chi tiết response
        logger.info(f"📤 [Outgoing Response - NonStream] User: {user_id}")
        logger.info(f"💬 AI Response: {result['ai_response']}")
        logger.info(f"🛍️ Product IDs: {result.get('product_ids', [])}")

        # Add to span (hiển thị trong Jaeger UI)
        span.set_attribute("chat.ai_response", result["ai_response"][:200])  # Giới hạn 200 ký tự
        span.set_attribute("chat.product_count", len(result.get("product_ids", [])))
        span.add_event(
            "💬 AI response generated",
            attributes={
                "ai_response_preview": result["ai_response"][:100],
                "product_count": len(result.get("product_ids", [])),
                "product_ids": str(result.get("product_ids", [])[:5]),  # First 5 IDs
            },
        )

        return {
            "status": "success",
            "ai_response": result["ai_response"],
            "product_ids": result.get("product_ids", []),
        }
    except Exception as e:
        logger.error(f"Error in fashion_qa_chat: {e}", exc_info=True)
        raise HTTPException(status_code=500, detail=str(e)) from e
