import asyncio
import os
import platform

if platform.system() == "Windows":
    print("🔧 Windows detected: Applying SelectorEventLoopPolicy globally...")
    asyncio.set_event_loop_policy(asyncio.WindowsSelectorEventLoopPolicy())

import logging

import uvicorn
from fastapi import FastAPI
from fastapi.middleware.cors import CORSMiddleware
from fastapi.staticfiles import StaticFiles

from api.chatbot_route import router as chatbot_router
from api.conservation_route import router as conservation_router
from common.langfuse_client import initialize_langfuse
from config import PORT

# Configure Logging
logging.basicConfig(
    level=logging.INFO,
    format="%(asctime)s [%(levelname)s] %(name)s: %(message)s",
    handlers=[logging.StreamHandler()],
)

logger = logging.getLogger(__name__)

# ==========================================
# 🔥 LANGFUSE INITIALIZATION
# ==========================================
if initialize_langfuse():
    logger.info("✅ Langfuse initialized successfully")
else:
    logger.warning("⚠️ Langfuse initialization failed or keys missing")

app = FastAPI(
    title="Contract AI Service",
    description="API for Contract AI Service",
    version="1.0.0",
)

logger.info("✅ Clerk Authentication middleware DISABLED (for testing)")
app.add_middleware(
    CORSMiddleware,
    allow_origins=["*"],
    allow_credentials=True,
    allow_methods=["*"],
    allow_headers=["*"],
)

app.include_router(conservation_router)
app.include_router(chatbot_router)


# --- MOCK API FOR LOAD TESTING ---
try:
    from api.mock_api_route import router as mock_router

    app.include_router(mock_router)
    print("✅ Mock API Router mounted at /mock")
except ImportError:
    print("⚠️ Mock Router not found, skipping...")

# ==========================================
# 🟢 ĐOẠN MOUNT STATIC HTML CỦA BRO ĐÂY 🟢
# ==========================================
try:
    static_dir = os.path.join(os.path.dirname(__file__), "static")
    if not os.path.exists(static_dir):
        os.makedirs(static_dir)
    # Mount thư mục static để chạy file index.html
    app.mount("/static", StaticFiles(directory=static_dir, html=True), name="static")
    print(f"✅ Static files mounted at /static (Dir: {static_dir})")
except Exception as e:
    print(f"⚠️ Failed to mount static files: {e}")


from fastapi.responses import RedirectResponse


@app.get("/")
async def root():
    return RedirectResponse(url="/static/index.html")


if __name__ == "__main__":
    print("=" * 60)
    print("🚀 Contract AI Service Starting...")
    print("=" * 60)
    print(f"📡 REST API: http://localhost:{PORT}")
    print(f"📡 Test Chatbot: http://localhost:{PORT}/static/index.html")
    print(f"📚 API Docs: http://localhost:{PORT}/docs")
    print("=" * 60)

    ENABLE_RELOAD = False
    print(f"⚠️  Hot reload: {ENABLE_RELOAD}")

    reload_dirs = ["common", "api", "agent"]

    if ENABLE_RELOAD:
        os.environ["PYTHONUNBUFFERED"] = "1"

    uvicorn.run(
        "server:app",
        host="0.0.0.0",
        port=PORT,
        reload=ENABLE_RELOAD,
        reload_dirs=reload_dirs,
        log_level="info",
    )
