"""
CANIFA Data Retrieval Tool - Tối giản cho Agentic Workflow.
Hỗ trợ Hybrid Search: Semantic (Vector) + Metadata Filter.
"""

import json
import logging
from decimal import Decimal

from langchain_core.tools import tool
from pydantic import BaseModel, Field

from agent.tools.product_search_helpers import build_starrocks_query
from common.starrocks_connection import StarRocksConnection

from langsmith import traceable

logger = logging.getLogger(__name__)


class DecimalEncoder(json.JSONEncoder):
    """Xử lý kiểu Decimal từ Database khi convert sang JSON."""

    def default(self, obj):
        if isinstance(obj, Decimal):
            return float(obj)
        return super().default(obj)


class SearchParams(BaseModel):
    """Cấu trúc tham số tìm kiếm mà Agent phải cung cấp, map trực tiếp với Database."""

    query: str | None = Field(
        None,
        description="Câu hỏi/mục đích tự do của user (đi chơi, dự tiệc, phỏng vấn,...) - dùng cho Semantic Search",
    )
    keywords: str | None = Field(
        None, description="Từ khóa kỹ thuật cụ thể (áo polo, quần jean,...) - dùng cho LIKE search"
    )
    internal_ref_code: str | None = Field(None, description="Mã sản phẩm (ví dụ: 1TS23S012)")
    product_color_code: str | None = Field(None, description="Mã màu sản phẩm (ví dụ: 1TS23S012-SK010)")
    product_line_vn: str | None = Field(None, description="Dòng sản phẩm (Áo phông, Quần short,...)")
    gender_by_product: str | None = Field(None, description="Giới tính: male, female")
    age_by_product: str | None = Field(None, description="Độ tuổi: adult, kids, baby, others")
    master_color: str | None = Field(None, description="Màu sắc chính (Đen/ Black, Trắng/ White,...)")
    material_group: str | None = Field(None, description="Nhóm chất liệu (Knit - Dệt Kim, Woven - Dệt Thoi,...)")
    season: str | None = Field(None, description="Mùa (Spring Summer, Autumn Winter)")
    style: str | None = Field(None, description="Phong cách (Basic Update, Fashion,...)")
    fitting: str | None = Field(None, description="Form dáng (Regular, Slim, Loose,...)")
    form_neckline: str | None = Field(None, description="Kiểu cổ (Crew Neck, V-neck,...)")
    form_sleeve: str | None = Field(None, description="Kiểu tay (Short Sleeve, Long Sleeve,...)")
    price_min: float | None = Field(None, description="Giá thấp nhất")
    price_max: float | None = Field(None, description="Giá cao nhất")
    action: str = Field("search", description="Hành động: 'search' (tìm kiếm) hoặc 'visual_search' (phân tích ảnh)")


@tool(args_schema=SearchParams)
@traceable(run_type="tool", name="data_retrieval_tool")
async def data_retrieval_tool(
    action: str = "search",
    query: str | None = None,
    keywords: str | None = None,
    internal_ref_code: str | None = None,
    product_color_code: str | None = None,
    product_line_vn: str | None = None,
    gender_by_product: str | None = None,
    age_by_product: str | None = None,
    master_color: str | None = None,
    material_group: str | None = None,
    season: str | None = None,
    style: str | None = None,
    fitting: str | None = None,
    form_neckline: str | None = None,
    form_sleeve: str | None = None,
    price_min: float | None = None,
    price_max: float | None = None,
) -> str:
    """
    Tìm kiếm sản phẩm CANIFA - Phân biệt rõ giữa Semantic Search và Metadata Filter.

    ⚠️ QUAN TRỌNG - KHI NÀO DÙNG GÌ:

    1️⃣ DÙNG 'query' (Semantic Search - Vector Embedding):
       ✅ Khi user hỏi về MỤC ĐÍCH, BỐI CẢNH, PHONG CÁCH SỐNG
       ✅ Câu hỏi trừu tượng, không có từ khóa sản phẩm rõ ràng
       ✅ Ví dụ:
          - "Tìm đồ đi biển mát mẻ" → query="đồ đi biển mát mẻ"
          - "Quần áo cho buổi hẹn hò" → query="trang phục hẹn hò lịch sự"
          - "Đồ mặc dự tiệc sang trọng" → query="trang phục dự tiệc sang trọng"
          - "Outfit cho mùa đông ấm áp" → query="trang phục mùa đông ấm áp"

    2️⃣ DÙNG 'keywords' + METADATA FILTERS (Exact Match):
       ✅ Khi user hỏi về THUỘC TÍNH CỤ THỂ của sản phẩm
       ✅ Có TÊN SẢN PHẨM rõ ràng (áo polo, quần jean, váy liền,...)
       ✅ Có GIÁ, MÀU SẮC, SIZE, MÃ SẢN PHẨM
       ✅ Ví dụ:
          - "Áo polo nam" → keywords="áo polo", gender_by_product="male"
          - "Quần jean nữ dưới 500k" → keywords="quần jean", gender_by_product="female", price_max=500000
          - "Áo thun đen giá rẻ" → keywords="áo thun", master_color="Đen", price_max=200000
          - "Sản phẩm 8TS24W001" → internal_ref_code="8TS24W001"
          - "Váy liền cho bé gái màu hồng" → keywords="váy liền", gender_by_product="female", age_by_product="others", master_color="Hồng"

    🚫 KHÔNG BAO GIỜ DÙNG 'query' CHO:
       - Câu hỏi về GIÁ (dưới 400k, từ 200k-500k, giá rẻ,...)
       - Câu hỏi về MÀU SẮC cụ thể (đen, trắng, đỏ,...)
       - Câu hỏi về TÊN SẢN PHẨM (áo polo, quần jean, váy liền,...)
       - Câu hỏi về MÃ SẢN PHẨM (8TS24W001, 1DS24C015,...)

    💡 KẾT HỢP CẢ HAI (Hybrid):
       Chỉ dùng khi câu hỏi vừa có BỐI CẢNH trừu tượng, vừa có THUỘC TÍNH cụ thể:
       - "Tìm áo khoác ấm áp cho mùa đông, giá dưới 1 triệu"
         → query="áo khoác ấm áp mùa đông", price_max=1000000

    📝 VÍ DỤ CHI TIẾT:

    Example 1 - Semantic Search (MỤC ĐÍCH):
       User: "Tìm đồ đi làm chuyên nghiệp"
       Tool: data_retrieval_tool(query="trang phục công sở chuyên nghiệp")

    Example 2 - Metadata Filter (THUỘC TÍNH):
       User: "Cho tôi xem áo polo nam dưới 400k"
       Tool: data_retrieval_tool(keywords="áo polo", gender_by_product="male", price_max=400000)

    Example 3 - Metadata Only (GIÁ + MÀU):
       User: "Quần short đen giá rẻ"
       Tool: data_retrieval_tool(keywords="quần short", master_color="Đen", price_max=300000)

    Example 4 - Exact Match (MÃ SẢN PHẨM):
       User: "Cho tôi thông tin sản phẩm 8TS24W001"
       Tool: data_retrieval_tool(internal_ref_code="8TS24W001")

    Example 5 - Hybrid (BỐI CẢNH + FILTER):
       User: "Tìm áo khoác ấm cho mùa đông, cho bé trai, từ 200k-500k"
       Tool: data_retrieval_tool(query="áo khoác ấm áp mùa đông", age_by_product="others", gender_by_product="male", price_min=200000, price_max=500000)
    """
    try:
        # 1. Log & Prepare Params
        # 1. Log & Prepare Params
        params = SearchParams(
            action=action,
            query=query,
            keywords=keywords,
            internal_ref_code=internal_ref_code,
            product_color_code=product_color_code,
            product_line_vn=product_line_vn,
            gender_by_product=gender_by_product,
            age_by_product=age_by_product,
            master_color=master_color,
            material_group=material_group,
            season=season,
            style=style,
            fitting=fitting,
            form_neckline=form_neckline,
            form_sleeve=form_sleeve,
            price_min=price_min,
            price_max=price_max,
        )
        _log_agent_call(params)

        # 2. Prepare Vector (Async) if needed
        query_vector = None
        if query:
            from common.embedding_service import create_embedding_async

            query_vector = await create_embedding_async(query)

        # 3. Execute Search (Async)
        sql = build_starrocks_query(params, query_vector=query_vector)
        db = StarRocksConnection()
        products = await db.execute_query_async(sql)

        if not products:
            return _handle_no_results(query, keywords)

        # 4. Format Results
        clean_products = _format_product_results(products)
        return json.dumps(
            {"status": "success", "count": len(clean_products), "products": clean_products},
            ensure_ascii=False,
            cls=DecimalEncoder,
        )

    except Exception as e:
        logger.error(f"Error in data_retrieval_tool: {e}")
        return json.dumps({"status": "error", "message": str(e)})


def _log_agent_call(params: SearchParams):
    """Log parameters for debugging."""
    filtered_params = {k: v for k, v in params.dict().items() if v is not None}
    logger.info(f"📋 Agent Tool Call - data_retrieval_tool: {json.dumps(filtered_params, ensure_ascii=False)}")


def _handle_no_results(query: str | None, keywords: str | None) -> str:
    """Return standardized no-results message."""
    logger.warning(f"No products found for search: query={query}, keywords={keywords}")
    return json.dumps(
        {
            "status": "no_results",
            "message": "Không tìm thấy sản phẩm nào phù hợp với yêu cầu. Vui lòng thử lại với từ khóa hoặc bộ lọc khác.",
        },
        ensure_ascii=False,
    )


def _format_product_results(products: list[dict]) -> list[dict]:
    """Filter and format product fields for the agent."""
    allowed_fields = {
        "internal_ref_code",
        "magento_ref_code",
        "product_color_code",
        "product_name",
        "color_code",
        "master_color",
        "product_color_name",
        "season_sale",
        "season",
        "style",
        "fitting",
        "size_scale",
        "graphic",
        "pattern",
        "weaving",
        "shape_detail",
        "form_neckline",
        "form_sleeve",
        "form_length",
        "form_waistline",
        "form_shoulderline",
        "material",
        "product_group",
        "product_line_vn",
        "unit_of_measure",
        "sale_price",
        "original_price",
        "material_group",
        "product_line_en",
        "age_by_product",
        "gender_by_product",
        "product_image_url",
        "description_text",
        "product_image_url_thumbnail",
        "product_web_url",
        "product_web_material",
    }
    return [{k: v for k, v in p.items() if k in allowed_fields} for p in products[:5]]
