"""
Agent Helper Functions
Các hàm tiện ích cho chat controller.
"""

import json
import logging
import uuid

from langchain_core.messages import HumanMessage, ToolMessage
from langchain_core.runnables import RunnableConfig

from common.conversation_manager import ConversationManager
from common.langfuse_client import get_callback_handler
from .models import AgentState

logger = logging.getLogger(__name__)


def extract_product_ids(messages: list) -> list[dict]:
    """
    Extract full product info from tool messages (data_retrieval_tool results).
    Returns list of product objects with: sku, name, price, sale_price, url, thumbnail_image_url.
    """
    products = []
    seen_skus = set()

    for msg in messages:
        if isinstance(msg, ToolMessage):
            try:
                # Tool result is JSON string
                tool_result = json.loads(msg.content)

                # Check if tool returned products (new format with "results" wrapper)
                if tool_result.get("status") == "success":
                    # Handle both direct "products" and nested "results" format
                    product_list = []
                    
                    if "results" in tool_result:
                        # New format: {"results": [{"products": [...]}]}
                        for result_item in tool_result["results"]:
                            product_list.extend(result_item.get("products", []))
                    elif "products" in tool_result:
                        # Legacy format: {"products": [...]}
                        product_list = tool_result["products"]
                    
                    for product in product_list:
                        sku = product.get("sku") or product.get("internal_ref_code")
                        if sku and sku not in seen_skus:
                            seen_skus.add(sku)

                            # Extract full product info (already parsed by tool)
                            product_obj = {
                                "sku": sku,
                                "name": product.get("name", ""),
                                "price": product.get("price", 0),
                                "sale_price": product.get("sale_price"),
                                "url": product.get("url", ""),
                                "thumbnail_image_url": product.get("thumbnail_image_url", ""),
                            }
                            products.append(product_obj)
            except (json.JSONDecodeError, KeyError, TypeError) as e:
                logger.debug(f"Could not parse tool message for products: {e}")
                continue

    return products


def parse_ai_response(ai_raw_content: str, all_products: list) -> tuple[str, list, str | None]:
    """
    Parse AI response từ LLM output và map SKUs với product data.
    
    Flow:
    - LLM trả về: {"ai_response": "...", "product_ids": ["SKU1", "SKU2"], "user_insight": "..."}
    - all_products: List products enriched từ tool messages
    - Map SKUs → enriched products
    
    Args:
        ai_raw_content: Raw content từ AI response
        all_products: Products extracted từ tool messages (đã có đầy đủ info)
        
    Returns:
        tuple: (ai_text_response, final_products, user_insight)
    """
    ai_text_response = ai_raw_content
    final_products = all_products  # Default: trả về tất cả products từ tool
    user_insight = None
    
    logger.info(f"🤖 Raw AI JSON: {ai_raw_content}")

    try:
        # Try to parse if it's a JSON string from LLM
        ai_json = json.loads(ai_raw_content)
        ai_text_response = ai_json.get("ai_response", ai_raw_content)
        explicit_skus = ai_json.get("product_ids", [])
        user_insight = ai_json.get("user_insight")
        
        if explicit_skus and isinstance(explicit_skus, list):
            # LLM trả về list SKUs → Map với products đã có
            # Build lookup dict từ all_products
            product_lookup = {p["sku"]: p for p in all_products if p.get("sku")}
            
            # Map SKUs → enriched products
            mapped_products = []
            for sku in explicit_skus:
                if isinstance(sku, str) and sku in product_lookup:
                    mapped_products.append(product_lookup[sku])
                elif isinstance(sku, dict):
                    # LLM có thể trả về dict (legacy) → giữ nguyên
                    mapped_products.append(sku)
            
            if mapped_products:
                final_products = mapped_products
            else:
                # If explicit SKUs provided but none found in DB, return empty list
                # This prevents showing unrelated products when AI hallucinates or references old/invalid SKUs
                final_products = []
            
    except (json.JSONDecodeError, TypeError):
        pass

    return ai_text_response, final_products, user_insight


def prepare_execution_context(query: str, user_id: str, history: list, images: list | None):
    """
    Prepare initial state and execution config for the graph run.
    
    Returns:
        tuple: (initial_state, exec_config)
    """
    initial_state: AgentState = {
        "user_query": HumanMessage(content=query),
        "messages": [HumanMessage(content=query)],
        "history": history,
        "user_id": user_id,
        "images_embedding": [],
        "ai_response": None,
    }
    run_id = str(uuid.uuid4())

    # Metadata for LangChain (tags for logging/filtering)
    metadata = {
        "run_id": run_id,
        "tags": "chatbot,production",
    }

    langfuse_handler = get_callback_handler()

    exec_config = RunnableConfig(
        configurable={
            "user_id": user_id,
            "transient_images": images or [],
            "run_id": run_id,
        },
        run_id=run_id,
        metadata=metadata,
        callbacks=[langfuse_handler] if langfuse_handler else [],
    )
    return initial_state, exec_config


async def handle_post_chat_async(
    memory: ConversationManager, 
    identity_key: str, 
    human_query: str, 
    ai_response: dict | None
):
    """
    Save chat history in background task after response is sent.
    Lưu AI response dưới dạng JSON string.
    """
    if ai_response:
        try:
            # Convert dict thành JSON string để lưu vào TEXT field
            ai_response_json = json.dumps(ai_response, ensure_ascii=False)
            await memory.save_conversation_turn(identity_key, human_query, ai_response_json)
            logger.debug(f"Saved conversation for identity_key {identity_key}")
        except Exception as e:
            logger.error(f"Failed to save conversation for identity_key {identity_key}: {e}", exc_info=True)
