"""
Middleware Module - Gom Auth + Rate Limit
Singleton Pattern cho cả 2 services
"""
from __future__ import annotations

import logging
from collections.abc import Callable
from typing import TYPE_CHECKING

from fastapi import HTTPException, Request, status
from starlette.middleware.base import BaseHTTPMiddleware

if TYPE_CHECKING:
    from fastapi import FastAPI

logger = logging.getLogger(__name__)


# =============================================================================
# CONFIGURATION
# =============================================================================

# Public endpoints - không cần auth
PUBLIC_PATHS = {
    "/",
    "/health",
    "/docs",
    "/openapi.json",
    "/redoc",
}

# Public path prefixes
PUBLIC_PATH_PREFIXES = [
    "/static",
    "/mock",
]


# =============================================================================
# AUTH MIDDLEWARE CLASS
# =============================================================================

class ClerkAuthMiddleware(BaseHTTPMiddleware):
    """
    Clerk Authentication Middleware
    
    Flow:
    1. Frontend gửi request với Authorization: Bearer <clerk_token>
    2. Middleware verify token và extract user_id
    3. Attach user info vào request.state.user và request.state.user_id
    4. Routes lấy trực tiếp từ request.state (không cần Depends)
    """

    async def dispatch(self, request: Request, call_next: Callable):
        path = request.url.path
        method = request.method

        # ✅ Allow OPTIONS requests (CORS preflight)
        if method == "OPTIONS":
            return await call_next(request)

        # Skip public endpoints
        if path in PUBLIC_PATHS:
            return await call_next(request)

        # Skip public path prefixes
        if any(path.startswith(prefix) for prefix in PUBLIC_PATH_PREFIXES):
            return await call_next(request)

        # ✅ Authentication Process
        try:
            auth_header = request.headers.get("Authorization")

            # ========== DEV MODE: Bypass auth ==========
            dev_user_id = request.headers.get("X-Dev-User-Id")
            if dev_user_id:
                logger.warning(f"⚠️ DEV MODE: Using X-Dev-User-Id={dev_user_id}")
                request.state.user = {"customer_id": dev_user_id}
                request.state.user_id = dev_user_id
                request.state.is_authenticated = True
                return await call_next(request)

            # --- TRƯỜNG HỢP 1: KHÔNG CÓ TOKEN -> GUEST ---
            if not auth_header or not auth_header.startswith("Bearer "):
                # Guest Mode (Không User ID, Không Auth)
                # logger.debug(f"ℹ️ Guest access (no token) for {path}")
                request.state.user = None
                request.state.user_id = None
                request.state.is_authenticated = False
                return await call_next(request)

            # --- TRƯỜNG HỢP 2: CÓ TOKEN -> GỌI CANIFA VERIFY ---
            token = auth_header.replace("Bearer ", "")
            
            # Import Lazy để tránh circular import nếu có
            from common.canifa_api import verify_canifa_token, extract_user_id_from_canifa_response

            try:
                # 1. Gọi API Canifa
                user_data = await verify_canifa_token(token)
                
                # 2. Lấy User ID
                user_id = await extract_user_id_from_canifa_response(user_data)
                
                if user_id:
                    # ✅ VERIFY THÀNH CÔNG -> USER VIP
                    request.state.user = user_data
                    request.state.user_id = user_id
                    request.state.token = token
                    request.state.is_authenticated = True
                    logger.debug(f"✅ Auth Success: User {user_id}")
                else:
                    # ❌ VERIFY FAILED -> GUEST
                    logger.warning(f"⚠️ Invalid Canifa Token (No ID found) -> Guest Mode")
                    request.state.user = None
                    request.state.user_id = None
                    request.state.is_authenticated = False
                    
            except Exception as e:
                logger.error(f"❌ Canifa Auth Error: {e} -> Guest Mode")
                request.state.user = None
                request.state.user_id = None
                request.state.is_authenticated = False

        except Exception as e:
            logger.error(f"❌ Middleware Unexpected Error: {e}")
            # Fallback an toàn: Guest mode
            request.state.user = None
            request.state.user_id = None
            request.state.is_authenticated = False
            
        return await call_next(request)


# =============================================================================
# MIDDLEWARE MANAGER - Singleton to manage all middlewares
# =============================================================================

class MiddlewareManager:
    """
    Middleware Manager - Singleton Pattern
    Quản lý và setup tất cả middlewares cho FastAPI app
    
    Usage:
        from common.middleware import middleware_manager
        
        app = FastAPI()
        middleware_manager.setup(app, enable_auth=True, enable_rate_limit=True)
    """
    
    _instance: MiddlewareManager | None = None
    _initialized: bool = False
    
    def __new__(cls) -> MiddlewareManager:
        if cls._instance is None:
            cls._instance = super().__new__(cls)
        return cls._instance
    
    def __init__(self) -> None:
        if MiddlewareManager._initialized:
            return
        
        self._auth_enabled = False
        self._rate_limit_enabled = False
        
        MiddlewareManager._initialized = True
        logger.info("✅ MiddlewareManager initialized")
    
    def setup(
        self,
        app: FastAPI,
        *,
        enable_auth: bool = True,
        enable_rate_limit: bool = True,
        enable_cors: bool = True,
        cors_origins: list[str] | None = None,
    ) -> None:
        """
        Setup tất cả middlewares cho FastAPI app.
        
        Args:
            app: FastAPI application
            enable_auth: Bật Clerk authentication middleware
            enable_rate_limit: Bật rate limiting
            enable_cors: Bật CORS middleware
            cors_origins: List origins cho CORS (default: ["*"])
        
        Note:
            Thứ tự middleware quan trọng! Middleware thêm sau sẽ chạy TRƯỚC.
            Order: CORS → Auth → RateLimit → SlowAPI
        """
        # 1. CORS Middleware (thêm cuối cùng để chạy đầu tiên)
        if enable_cors:
            self._setup_cors(app, cors_origins or ["*"])
        
        # 2. Auth Middleware
        if enable_auth:
            self._setup_auth(app)
        
        # 3. Rate Limit Middleware
        if enable_rate_limit:
            self._setup_rate_limit(app)
        
        logger.info(
            f"✅ Middlewares configured: "
            f"CORS={enable_cors}, Auth={enable_auth}, RateLimit={enable_rate_limit}"
        )
    
    def _setup_cors(self, app: FastAPI, origins: list[str]) -> None:
        """Setup CORS middleware."""
        from fastapi.middleware.cors import CORSMiddleware
        
        app.add_middleware(
            CORSMiddleware,
            allow_origins=origins,
            allow_credentials=True,
            allow_methods=["*"],
            allow_headers=["*"],
        )
        logger.info(f"✅ CORS middleware enabled (origins: {origins})")
    
    def _setup_auth(self, app: FastAPI) -> None:
        """Setup Clerk auth middleware."""
        app.add_middleware(ClerkAuthMiddleware)
        self._auth_enabled = True
        logger.info("✅ Clerk Auth middleware enabled")
    
    def _setup_rate_limit(self, app: FastAPI) -> None:
        """Setup rate limiting."""
        from common.rate_limit import rate_limit_service
        
        rate_limit_service.setup(app)
        self._rate_limit_enabled = True
        logger.info("✅ Rate Limit middleware enabled")
    
    @property
    def is_auth_enabled(self) -> bool:
        return self._auth_enabled
    
    @property
    def is_rate_limit_enabled(self) -> bool:
        return self._rate_limit_enabled


# =============================================================================
# SINGLETON INSTANCE
# =============================================================================

middleware_manager = MiddlewareManager()
