"""
Fashion Q&A Agent Router
FastAPI endpoints cho Fashion Q&A Agent service.
Router chỉ chứa định nghĩa API, logic nằm ở controller.

Message Limit:
- Guest (không login): 10 tin/ngày theo device_id
- User đã login: 100 tin/ngày theo user_id
"""

import logging

from fastapi import APIRouter, BackgroundTasks, HTTPException, Request
from opentelemetry import trace

from agent.controller import chat_controller
from agent.models import QueryRequest
from common.message_limit import message_limit_service
from common.user_identity import get_user_identity
from config import DEFAULT_MODEL

logger = logging.getLogger(__name__)
tracer = trace.get_tracer(__name__)
router = APIRouter()


@router.post("/api/agent/chat", summary="Fashion Q&A Chat (Non-streaming)")
async def fashion_qa_chat(request: Request, req: QueryRequest, background_tasks: BackgroundTasks):
    """
    Endpoint chat không stream - trả về response JSON đầy đủ một lần.
    
    Message Limit:
    - Guest: 10 tin nhắn/ngày (theo device_id)
    - User đã login: 100 tin nhắn/ngày (theo user_id)
    """
    # 1. Xác định user identity
    identity = get_user_identity(request)
    user_id = identity.primary_id
    
    # 2. Check message limit TRƯỚC khi xử lý
    can_send, limit_info = await message_limit_service.check_limit(
        identity_key=identity.rate_limit_key,
        is_authenticated=identity.is_authenticated,
    )
    
    if not can_send:
        logger.warning(
            f"⚠️ Message limit exceeded: {identity.rate_limit_key} | "
            f"used={limit_info['used']}/{limit_info['limit']}"
        )
        return {
            "status": "error",
            "error_code": "MESSAGE_LIMIT_EXCEEDED",
            "message": limit_info["message"],
            "require_login": limit_info["require_login"],
            "limit_info": {
                "limit": limit_info["limit"],
                "used": limit_info["used"],
                "remaining": limit_info["remaining"],
                "reset_seconds": limit_info["reset_seconds"],
            },
        }

    logger.info(f"📥 [Incoming Query - NonStream] User: {user_id} | Query: {req.user_query}")

    # Get current span để add logs VÀO JAEGER UI
    span = trace.get_current_span()
    span.set_attribute("user.id", user_id)
    span.set_attribute("chat.user_query", req.user_query)
    span.add_event(
        "📥 User query received", attributes={"user_id": user_id, "query": req.user_query, "timestamp": "incoming"}
    )

    try:
        # Gọi controller để xử lý logic (Non-streaming)
        result = await chat_controller(
            query=req.user_query,
            user_id=user_id,
            background_tasks=background_tasks,
            model_name=DEFAULT_MODEL,
            images=req.images,
        )

        # Log chi tiết response
        logger.info(f"📤 [Outgoing Response - NonStream] User: {user_id}")
        logger.info(f"💬 AI Response: {result['ai_response']}")
        logger.info(f"🛍️ Product IDs: {result.get('product_ids', [])}")

        # Add to span (hiển thị trong Jaeger UI)
        span.set_attribute("chat.ai_response", result["ai_response"][:200])  # Giới hạn 200 ký tự
        span.set_attribute("chat.product_count", len(result.get("product_ids", [])))
        span.add_event(
            "💬 AI response generated",
            attributes={
                "ai_response_preview": result["ai_response"][:100],
                "product_count": len(result.get("product_ids", [])),
                "product_ids": str(result.get("product_ids", [])[:5]),  # First 5 IDs
            },
        )

        # 3. Increment message count SAU KHI chat thành công
        usage_info = await message_limit_service.increment(
            identity_key=identity.rate_limit_key,
            is_authenticated=identity.is_authenticated,
        )

        return {
            "status": "success",
            "ai_response": result["ai_response"],
            "product_ids": result.get("product_ids", []),
            "limit_info": {
                "limit": usage_info["limit"],
                "used": usage_info["used"],
                "remaining": usage_info["remaining"],
            },
        }
    except Exception as e:
        logger.error(f"Error in fashion_qa_chat: {e}", exc_info=True)
        raise HTTPException(status_code=500, detail=str(e)) from e

