"""
Fashion Q&A Agent Controller
Langfuse will auto-trace via LangChain integration (no code changes needed).
"""

import json
import logging
import time
import uuid

from fastapi import BackgroundTasks
from langchain_core.messages import AIMessage, HumanMessage, ToolMessage
from langchain_core.runnables import RunnableConfig

from common.conversation_manager import ConversationManager, get_conversation_manager
from common.langfuse_client import get_callback_handler
from common.llm_factory import create_llm
from config import DEFAULT_MODEL
from langfuse import propagate_attributes

from .graph import build_graph
from .models import AgentState, get_config
from .tools.get_tools import get_all_tools

logger = logging.getLogger(__name__)


async def chat_controller(
    query: str,
    user_id: str,
    background_tasks: BackgroundTasks,
    model_name: str = DEFAULT_MODEL,
    images: list[str] | None = None,
) -> dict:
    """
    Controller main logic for non-streaming chat requests.

    Tạm thời bỏ lớp cache để đơn giản luồng xử lý:
    - Nhận query → Gọi LLM qua graph.
    - Lưu lịch sử hội thoại ở background.
    """
    logger.info("chat_controller start: model=%s, user_id=%s", model_name, user_id)

    # ====================== CACHE LAYER (TẠM THỜI TẮT) ======================
    # from common.cache import redis_cache
    #
    # cached_response = await redis_cache.get_response(user_id=user_id, query=query)
    # if cached_response:
    #     # CACHE HIT - Return immediately
    #     memory = await get_conversation_manager()
    #     background_tasks.add_task(
    #         _handle_post_chat_async,
    #         memory=memory,
    #         user_id=user_id,
    #         human_query=query,
    #         ai_msg=AIMessage(content=cached_response["ai_response"]),
    #     )
    #     return {**cached_response, "cached": True}

    # ====================== NORMAL LLM FLOW ======================
    logger.info("chat_controller: proceed with live LLM call")

    config = get_config()
    config.model_name = model_name

    # Enable JSON mode to ensure structured output
    llm = create_llm(model_name=model_name, streaming=False, json_mode=True)
    tools = get_all_tools()

    graph = build_graph(config, llm=llm, tools=tools)

    # Init ConversationManager (Singleton)
    memory = await get_conversation_manager()

    # LOAD HISTORY & Prepare State
    history_dicts = await memory.get_chat_history(user_id, limit=20)
    messages = []
    for m in history_dicts:
        if m["is_human"]: # Original code used 'is_human', new code used 'role'
            messages.append(HumanMessage(content=m["message"]))
        else:
            messages.append(AIMessage(content=m["message"]))

    # Prepare initial state and execution config for the graph run.
    initial_state: AgentState = {
        "user_query": HumanMessage(content=query),
        "messages": messages + [HumanMessage(content=query)],
        "history": messages, # The new code uses 'messages' for history, which is correct
        "user_id": user_id,
        "images_embedding": [],
        "ai_response": None,
    }
    run_id = str(uuid.uuid4())

    # Metadata for LangChain (tags for logging/filtering)
    metadata = {
        "run_id": run_id,
        "tags": "chatbot,production",
    }

    langfuse_handler = get_callback_handler()

    exec_config = RunnableConfig(
        configurable={
            "user_id": user_id,
            "transient_images": images or [],
            "run_id": run_id,
        },
        run_id=run_id,
        metadata=metadata,
        callbacks=[langfuse_handler] if langfuse_handler else [],
    )

    # Execute graph với Langfuse user tracking
    # Dùng propagate_attributes để tự động gán user_id cho tất cả observations
    start_time = time.time()
    # Generate session_id từ user_id + run_id (có thể thay bằng conversation_id nếu có)
    session_id = f"{user_id}-{run_id[:8]}"
    
    with propagate_attributes(user_id=user_id, session_id=session_id):
        result = await graph.ainvoke(initial_state, config=exec_config)
    duration = time.time() - start_time

    # Parse AI response (expected JSON from chat_controller logic)
    all_product_ids = _extract_product_ids(result.get("messages", []))
    ai_raw_content = result.get("ai_response").content if result.get("ai_response") else ""
    logger.debug("raw ai output: %s", ai_raw_content)

    # Standardize output
    ai_text_response = ai_raw_content
    final_product_ids = all_product_ids

    try:
        # Try to parse if it's a JSON string from LLM
        ai_json = json.loads(ai_raw_content)
        ai_text_response = ai_json.get("ai_response", ai_raw_content)
        explicit_ids = ai_json.get("product_ids", [])
        if explicit_ids and isinstance(explicit_ids, list):
            # Merge with extracted IDs if needed or replace
            final_product_ids = explicit_ids
    except:
        pass

    response_payload = {
        "ai_response": ai_text_response,
        "product_ids": final_product_ids,
    }

    # ====================== STORE LAYER 1 CACHE (TẠM THỜI TẮT) ======================
    # Cache for 5 minutes (300s) - Short enough for stock safety
    # await redis_cache.set_response(user_id=user_id, query=query, response_data=response_payload, ttl=300)

    # Add to history in background - lưu nguyên response JSON
    background_tasks.add_task(
        _handle_post_chat_async,
        memory=memory,
        user_id=user_id,
        human_query=query,
        ai_response=response_payload,  # dict: {ai_response, product_ids}
    )

    logger.info("chat_controller finished in %.2fs", duration)
    return {**response_payload, "cached": False}


def _extract_product_ids(messages: list) -> list[dict]:
    """
    Extract full product info from tool messages (data_retrieval_tool results).
    Returns list of product objects with: sku, name, price, sale_price, url, thumbnail_image_url.
    """
    products = []
    seen_skus = set()

    for msg in messages:
        if isinstance(msg, ToolMessage):
            try:
                # Tool result is JSON string
                tool_result = json.loads(msg.content)

                # Check if tool returned products
                if tool_result.get("status") == "success" and "products" in tool_result:
                    for product in tool_result["products"]:
                        sku = product.get("internal_ref_code")
                        if sku and sku not in seen_skus:
                            seen_skus.add(sku)

                            # Extract full product info
                            product_obj = {
                                "sku": sku,
                                "name": product.get("magento_product_name", ""),
                                "price": product.get("price_vnd", 0),
                                "sale_price": product.get("sale_price_vnd"),  # null nếu không sale
                                "url": product.get("magento_url_key", ""),
                                "thumbnail_image_url": product.get("thumbnail_image_url", ""),
                            }
                            products.append(product_obj)
            except (json.JSONDecodeError, KeyError, TypeError) as e:
                logger.debug(f"Could not parse tool message for products: {e}")
                continue

    return products


def _prepare_execution_context(query: str, user_id: str, history: list, images: list | None):
    """Prepare initial state and execution config for the graph run."""
    initial_state: AgentState = {
        "user_query": HumanMessage(content=query),
        "messages": [HumanMessage(content=query)],
        "history": history,
        "user_id": user_id,
        "images_embedding": [],
        "ai_response": None,
    }
    run_id = str(uuid.uuid4())

    # Metadata for LangChain (tags for logging/filtering)
    metadata = {
        "run_id": run_id,
        "tags": "chatbot,production",
    }

    langfuse_handler = get_callback_handler()

    exec_config = RunnableConfig(
        configurable={
            "user_id": user_id,
            "transient_images": images or [],
            "run_id": run_id,
        },
        run_id=run_id,
        metadata=metadata,
        callbacks=[langfuse_handler] if langfuse_handler else [],
    )
    return initial_state, exec_config


async def _handle_post_chat_async(
    memory: ConversationManager, user_id: str, human_query: str, ai_response: dict | None
):
    """
    Save chat history in background task after response is sent.
    Lưu AI response dưới dạng JSON string.
    """
    if ai_response:
        try:
            # Convert dict thành JSON string để lưu vào TEXT field
            ai_response_json = json.dumps(ai_response, ensure_ascii=False)
            await memory.save_conversation_turn(user_id, human_query, ai_response_json)
            logger.debug(f"Saved conversation for user {user_id}")
        except Exception as e:
            logger.error(f"Failed to save conversation for user {user_id}: {e}", exc_info=True)
