import logging
from datetime import datetime

from langchain_core.tools import tool
from pydantic import BaseModel, Field

from common.starrocks_connection import get_db_connection
from agent.prompt_utils import read_tool_prompt

logger = logging.getLogger(__name__)


class PromotionInput(BaseModel):
    check_date: str | None = Field(
        ...,
        description="Ngày cần kiểm tra khuyến mãi (định dạng YYYY-MM-DD). Nếu muốn kiểm tra ngày hiện tại thì truyền null."
    )
    model_config = {"extra": "forbid"}


@tool("canifa_get_promotions", args_schema=PromotionInput)
async def canifa_get_promotions(check_date: str = None) -> str:
    """
    Tra cứu danh sách các chương trình khuyến mãi (CTKM) đang diễn ra theo ngày.
    Sử dụng tool này khi khách hàng hỏi về:
    - "Hôm nay có khuyến mãi gì không?"
    - "Đang có chương trình gì hot?"
    - "Ngày mai có giảm giá không?"
    - "Danh sách mã giảm giá hiện tại."

    Trả về: Tên chương trình, mô tả, thời gian áp dụng.
    """
    target_date = check_date
    if not target_date:
        target_date = datetime.now().strftime("%Y-%m-%d")
    
    logger.info(f"🎁 [Promotion Search] Checking for date: {target_date}")

    try:
        sql = f"""
        SELECT 
            rule_id,
            name,
            description,
            from_date,
            to_date
        FROM shared_source.magento_salesrule
        WHERE '{target_date}' >= DATE(from_date) 
          AND '{target_date}' <= DATE(to_date)
        ORDER BY to_date ASC
        LIMIT 10
        """

        sr = get_db_connection()
        results = await sr.execute_query_async(sql)

        if not results:
            return f"Hiện tại (ngày {target_date}) không có chương trình khuyến mãi nào đang diễn ra trên hệ thống."

        lines = []
        for res in results:
            name = res.get("name", "CTKM")
            desc = res.get("description", "")
            f_date = res.get("from_date", "")
            t_date = res.get("to_date", "")
            
            lines.append(f"- **{name}**\n  {desc}\n  (Từ {f_date} đến {t_date})")

        return "\n\n".join(lines)

    except Exception as e:
        logger.error(f"❌ Error in canifa_get_promotions: {e}")
        return "Xin lỗi, tôi không thể lấy danh sách khuyến mãi lúc này."

# Load dynamic docstring
canifa_get_promotions.__doc__ = read_tool_prompt("promotion_canifa_tool") or canifa_get_promotions.__doc__
