"""
Fashion Q&A Agent Controller
Langfuse will auto-trace via LangChain integration (no code changes needed).
"""

import logging
import uuid

from fastapi import BackgroundTasks
from langchain_core.messages import AIMessage, HumanMessage
from langchain_core.runnables import RunnableConfig

from common.cache import redis_cache
from common.conversation_manager import get_conversation_manager
from common.langfuse_client import get_callback_handler
from config import DEFAULT_MODEL, REDIS_CACHE_TURN_ON
from langfuse import propagate_attributes

from .graph import build_graph
from .helper import extract_product_ids, handle_post_chat_async, parse_ai_response
from .models import AgentState, get_config

logger = logging.getLogger(__name__)


async def chat_controller(
    query: str,
    user_id: str,
    background_tasks: BackgroundTasks,
    model_name: str = DEFAULT_MODEL,
    images: list[str] | None = None,
    identity_key: str | None = None,
) -> dict:
    """
    Controller main logic for non-streaming chat requests.

    Flow:
    1. Check cache (if enabled) → HIT: return cached response
    2. MISS: Call LLM → Save to cache → Return response
    
    Args:
        identity_key: Key for saving/loading history (identity.history_key)
                      Guest: device_id, User: user_id
    """
    effective_identity_key = identity_key or user_id
    
    logger.info(
        "chat_controller start: model=%s, user_id=%s, identity_key=%s", 
        model_name, user_id, effective_identity_key
    )

    # ====================== CACHE LAYER ======================
    if REDIS_CACHE_TURN_ON:
        cached_response = await redis_cache.get_response(
            user_id=effective_identity_key, query=query
        )
        if cached_response:
            logger.info(f"⚡ CACHE HIT for identity_key={effective_identity_key}")
            memory = await get_conversation_manager()
            background_tasks.add_task(
                handle_post_chat_async,
                memory=memory,
                identity_key=effective_identity_key,
                human_query=query,
                ai_response=cached_response,
            )
            return {**cached_response, "cached": True}

    # ====================== NORMAL LLM FLOW ======================
    logger.info("chat_controller: proceed with live LLM call")

    config = get_config()
    config.model_name = model_name

    graph = build_graph(config)

    # Init ConversationManager (Singleton)
    memory = await get_conversation_manager()

    # Load History (only text, no product_ids for AI context)
    history_dicts = await memory.get_chat_history(effective_identity_key, limit=15, include_product_ids=False)
    messages = [
        HumanMessage(content=m["message"]) if m["is_human"] else AIMessage(content=m["message"])
        for m in history_dicts
    ][::-1]  # Reverse to chronological order (Oldest -> Newest)

    # Prepare State
    # Fetch User Insight from Redis
    user_insight = None
    if effective_identity_key:
        try:
            client = redis_cache.get_client()
            if client:
                insight_key = f"identity_key_insight:{effective_identity_key}"
                user_insight = await client.get(insight_key)
                if user_insight:
                    logger.info(f"🧠 Loaded User Insight for {effective_identity_key}: {user_insight[:50]}...")
        except Exception as e:
            logger.error(f"❌ Error fetching user insight: {e}")

    initial_state: AgentState = {
        "user_query": HumanMessage(content=query),
        "messages": [],  # Start empty, acting as scratchpad. History & Query are separate.
        "history": messages,
        "user_id": user_id,
        "user_insight": user_insight,
        "images_embedding": [],
        "ai_response": None,
    }
    
    run_id = str(uuid.uuid4())
    langfuse_handler = get_callback_handler()
    
    exec_config = RunnableConfig(
        configurable={
            "user_id": user_id,
            "transient_images": images or [],
            "run_id": run_id,
        },
        run_id=run_id,
        metadata={"run_id": run_id, "tags": "chatbot,production"},
        callbacks=[langfuse_handler] if langfuse_handler else [],
    )

    # Execute Graph
    session_id = f"{user_id}-{run_id[:8]}"
    
    with propagate_attributes(user_id=user_id, session_id=session_id):
        result = await graph.ainvoke(initial_state, config=exec_config)

    # Parse Response
    all_product_ids = extract_product_ids(result.get("messages", []))
    ai_raw_content = result.get("ai_response").content if result.get("ai_response") else ""
    # Unpack 3 values now
    ai_text_response, final_product_ids, new_insight = parse_ai_response(ai_raw_content, all_product_ids)

    # Save new insight to Redis if available
    if new_insight and effective_identity_key:
        try:
            client = redis_cache.get_client()
            if client:
                insight_key = f"identity_key_insight:{effective_identity_key}"
                # Save insight without TTL (or long TTL) as it is persistent user data
                await client.set(insight_key, new_insight)
                logger.info(f"💾 Updated User Insight for {effective_identity_key}: {new_insight}")
        except Exception as e:
            logger.error(f"❌ Failed to save user insight: {e}")

    response_payload = {
        "ai_response": ai_text_response,
        "product_ids": final_product_ids,
        "user_insight": new_insight,  # Return for dev/debug purposes
    }

    # ====================== SAVE TO CACHE ======================
    if REDIS_CACHE_TURN_ON:
        await redis_cache.set_response(
            user_id=effective_identity_key, 
            query=query, 
            response_data=response_payload, 
            ttl=300
        )
        logger.debug(f"💾 Cached response for identity_key={effective_identity_key}")

    # Save to History (Background) - We save the payload WITHOUT insight to history text
    background_tasks.add_task(
        handle_post_chat_async,
        memory=memory,
        identity_key=effective_identity_key,
        human_query=query,
        ai_response=response_payload,
    )

    logger.info("chat_controller finished")
    return {**response_payload, "cached": False}

