
import logging
import httpx
from langchain_core.tools import tool
from pydantic import BaseModel, Field

logger = logging.getLogger(__name__)

class StockCheckInput(BaseModel):
    skus: str = Field(
        description="Danh sách mã SKU sản phẩm cần kiểm tra tồn kho, phân cách bằng dấu phẩy. Ví dụ: '6ST25W005-SE091-L,6ST25W005-SE091-M'"
    )

@tool("check_is_stock", args_schema=StockCheckInput)
async def check_is_stock(skus: str) -> str:
    """
    Kiểm tra tình trạng tồn kho của các mã sản phẩm (SKU) thực tế từ hệ thống Canifa.
    Sử dụng tool này khi người dùng hỏi về tình trạng còn hàng, hết hàng của sản phẩm cụ thể.
    Input nhận vào là chuỗi các SKU phân cách bởi dấu phẩy.
    """
    logger.info(f"🔍 [Stock Check] Checking stock for SKUs: {skus}")
    
    url = "https://canifa.com/v1/middleware/stock_get_stock_list"
    params = {"skus": skus}
    
    try:
        async with httpx.AsyncClient() as client:
            response = await client.get(url, params=params, timeout=10.0)
            response.raise_for_status()
            
            data = response.json()
            logger.info(f"✅ Stock Check response: {str(data)[:200]}...")
            
            # Trả về raw JSON để LLM tự xử lý thông tin
            return str(data)
            
    except httpx.RequestError as e:
        logger.error(f"❌ Network error checking stock: {e}")
        return f"Lỗi kết nối khi kiểm tra tồn kho: {str(e)}"
    except httpx.HTTPStatusError as e:
        logger.error(f"❌ HTTP error {e.response.status_code}: {e}")
        return f"Lỗi server khi kiểm tra tồn kho (Status {e.response.status_code})"
    except Exception as e:
        logger.error(f"❌ Unexpected error in check_is_stock: {e}")
        return f"Lỗi không xác định khi kiểm tra tồn kho: {str(e)}"
