"""
Langfuse Client với OpenInference instrumentation cho Agno
Tự động trace tất cả Agno calls (LLM, tools, agent runs)
"""

import asyncio
import base64
import logging
import os
from concurrent.futures import ThreadPoolExecutor
from contextlib import contextmanager

from langfuse import Langfuse, get_client, propagate_attributes
from langfuse.langchain import CallbackHandler

from config import (
    LANGFUSE_BASE_URL,
    LANGFUSE_PUBLIC_KEY,
    LANGFUSE_SECRET_KEY,
)

# OpenInference imports (optional - only if available)
_OPENINFERENCE_AVAILABLE = False
AgnoInstrumentor = None  # type: ignore

try:
    from openinference.instrumentation.agno import AgnoInstrumentor  # type: ignore[import-untyped]
    from opentelemetry import trace as trace_api
    from opentelemetry.exporter.otlp.proto.http.trace_exporter import OTLPSpanExporter
    from opentelemetry.sdk.trace import TracerProvider
    from opentelemetry.sdk.trace.export import SimpleSpanProcessor

    _OPENINFERENCE_AVAILABLE = True
except ImportError:
    pass

logger = logging.getLogger(__name__)

# ⚡ Global state for async batch export
_langfuse_client: Langfuse | None = None
_export_executor: ThreadPoolExecutor | None = None
_pending_traces: list = []
_export_task: asyncio.Task | None = None
_batch_lock = asyncio.Lock if hasattr(asyncio, "Lock") else None


def initialize_langfuse() -> bool:
    """
    1. Setup OpenInference instrumentation cho Agno (nếu available)
    2. Configure OTLP exporter để gửi traces đến Langfuse
    3. Initialize Langfuse client (fallback)
    4. Register shutdown handler
    """
    global _langfuse_client, _export_executor

    if not LANGFUSE_PUBLIC_KEY or not LANGFUSE_SECRET_KEY:
        logger.warning("⚠️ LANGFUSE KEYS MISSING. Tracing disabled.")
        return False

    # Set environment
    os.environ["LANGFUSE_PUBLIC_KEY"] = LANGFUSE_PUBLIC_KEY
    os.environ["LANGFUSE_SECRET_KEY"] = LANGFUSE_SECRET_KEY
    base_url = LANGFUSE_BASE_URL or "https://cloud.langfuse.com"
    os.environ["LANGFUSE_BASE_URL"] = base_url
    os.environ["LANGFUSE_TIMEOUT"] = "10"
    os.environ["LANGFUSE_FLUSHINTERVAL"] = "300"

    try:
        # ========== Setup OpenInference cho Agno ==========
        global _OPENINFERENCE_AVAILABLE
        if _OPENINFERENCE_AVAILABLE:
            try:
                # Determine Langfuse OTLP endpoint
                if "localhost" in base_url or "127.0.0.1" in base_url:
                    otlp_endpoint = f"{base_url}/api/public/otel"
                elif "us.cloud" in base_url:
                    otlp_endpoint = "https://us.cloud.langfuse.com/api/public/otel"
                elif "eu.cloud" in base_url:
                    otlp_endpoint = "https://eu.cloud.langfuse.com/api/public/otel"
                else:
                    # Custom deployment
                    otlp_endpoint = f"{base_url}/api/public/otel"

                # Create auth header
                langfuse_auth = base64.b64encode(
                    f"{LANGFUSE_PUBLIC_KEY}:{LANGFUSE_SECRET_KEY}".encode()
                ).decode()

                # Set OTLP environment variables
                os.environ["OTEL_EXPORTER_OTLP_ENDPOINT"] = otlp_endpoint
                os.environ["OTEL_EXPORTER_OTLP_HEADERS"] = f"Authorization=Basic {langfuse_auth}"

                # Configure TracerProvider
                tracer_provider = TracerProvider()
                tracer_provider.add_span_processor(SimpleSpanProcessor(OTLPSpanExporter()))
                trace_api.set_tracer_provider(tracer_provider=tracer_provider)

                # Instrument Agno
                if AgnoInstrumentor:
                    AgnoInstrumentor().instrument()

                logger.info(f"✅ OpenInference instrumentation enabled for Agno")
                logger.info(f"   → Sending traces to: {otlp_endpoint}")
            except Exception as e:
                logger.warning(f"⚠️ Failed to setup OpenInference: {e}. Falling back to Langfuse SDK.")
                _OPENINFERENCE_AVAILABLE = False

        # ========== Fallback: Langfuse SDK ==========
        if not _OPENINFERENCE_AVAILABLE:
            _langfuse_client = get_client()
            _export_executor = ThreadPoolExecutor(max_workers=1, thread_name_prefix="langfuse_export")

        # Register shutdown handler
        import atexit
        atexit.register(shutdown_langfuse)

        logger.info(f"✅ Langfuse initialized (BASE_URL: {base_url})")
        return True

    except Exception as e:
        logger.error(f"❌ Langfuse init error: {e}")
        return False


def shutdown_langfuse():
    """Shutdown Langfuse client gracefully để tránh nghẽn khi exit"""
    global _langfuse_client, _export_executor

    try:
        if _langfuse_client:
            # Flush pending traces trước khi shutdown
            try:
                _langfuse_client.flush()
            except Exception as e:
                logger.debug(f"Langfuse flush error during shutdown: {e}")

            # Shutdown client (non-blocking với timeout)
            try:
                if hasattr(_langfuse_client, "shutdown"):
                    _langfuse_client.shutdown()
            except Exception as e:
                logger.debug(f"Langfuse shutdown error: {e}")

        if _export_executor:
            _export_executor.shutdown(wait=False)  # Non-blocking shutdown

        logger.debug("🔒 Langfuse client shutdown completed")
    except Exception as e:
        logger.debug(f"Error during Langfuse shutdown: {e}")


async def async_flush_langfuse():
    """
    Async wrapper to flush Langfuse without blocking event loop.
    Uses thread pool executor to run sync flush in background.
    """
    if not _langfuse_client or not _export_executor:
        return

    try:
        loop = asyncio.get_event_loop()
        # Run flush in thread pool (non-blocking)
        await loop.run_in_executor(_export_executor, _langfuse_client.flush)
        logger.debug("📤 Langfuse flushed (async)")
    except Exception as e:
        logger.warning(f"⚠️ Async flush failed: {e}")


def get_callback_handler(
    trace_id: str | None = None,
    user_id: str | None = None,
    session_id: str | None = None,
    tags: list[str] | None = None,
    **trace_kwargs,
) -> CallbackHandler | None:
    """
    Get CallbackHandler with unique trace context.

    Args:
        trace_id: Optional unique trace ID
        user_id: User ID for grouping traces by user (NOT set here - use propagate_attributes instead)
        session_id: Session ID for grouping traces by session/conversation
        tags: List of tags for filtering traces
        **trace_kwargs: Additional trace attributes

    Returns:
        CallbackHandler instance + propagate_attributes context manager

    Note:
        Per Langfuse docs: use propagate_attributes(user_id=...) context manager
        to properly set user_id across all observations in the trace.
        This makes user_id appear as a filterable field in Langfuse UI.
    """
    try:
        if not _langfuse_client:
            logger.warning("⚠️ Langfuse client not initialized")
            return None

        handler = CallbackHandler()
        logger.debug("✅ Langfuse CallbackHandler created")
        return handler
    except Exception as e:
        logger.warning(f"⚠️ CallbackHandler error: {e}")
        return None


@contextmanager
def langfuse_trace_context(user_id: str | None = None, session_id: str | None = None, tags: list[str] | None = None):
    """
    Context manager to propagate user_id, session_id, tags to all observations.

    Usage:
        with langfuse_trace_context(user_id="user_123", session_id="session_456"):
            # All observations created here will have these attributes
            await invoke_chain()
    """
    attrs = {}
    if user_id:
        attrs["user_id"] = user_id
    if session_id:
        attrs["session_id"] = session_id

    # Tags are set via metadata, not propagate_attributes
    with propagate_attributes(**attrs):
        yield
