"""
CANIFA Agent Controller với Agno Framework
"""

import json
import logging
from typing import Any

from fastapi import BackgroundTasks

from common.langfuse_client import langfuse_trace_context
from config import DEFAULT_MODEL

from .agno_agent import get_agno_agent

logger = logging.getLogger(__name__)


async def chat_controller(
    query: str,
    user_id: str,
    background_tasks: BackgroundTasks,
    model_name: str = DEFAULT_MODEL,
    images: list[str] | None = None,
) -> dict:
    """
    Controller với Agno Agent (có memory tự động).
    Agno tự động load/save history qua ConversationManager.
    """
    logger.info(f"▶️ Agno chat_controller | User: {user_id} | Model: {model_name}")

    try:
        agent = await get_agno_agent(model_name=model_name, json_mode=True)

        with langfuse_trace_context(user_id=user_id, session_id=user_id):
            # Agno tự động load history và save sau khi respond (memory enabled)
            result = agent.run(query, session_id=user_id)

            # Extract response
            ai_content = str(result.content if hasattr(result, "content") and result.content else str(result))
            logger.info(f"💾 AI Response: {ai_content[:200]}...")

            # Parse response và extract products
            ai_text, product_ids = _parse_agno_response(result, ai_content)

            return {
                "ai_response": ai_text,
                "product_ids": product_ids,
            }

    except Exception as e:
        logger.error(f"💥 Agno chat error for user {user_id}: {e}", exc_info=True)
        raise


def _parse_agno_response(result: Any, ai_content: str) -> tuple[str, list[dict]]:
    """
    Parse Agno response và extract AI text + product IDs.
    Returns: (ai_text_response, product_ids)
    """
    ai_text = ai_content
    product_ids = []

    # Try parse JSON response
    try:
        ai_json = json.loads(ai_content)
        ai_text = ai_json.get("ai_response", ai_content)
        product_ids = ai_json.get("product_ids", []) or []
    except (json.JSONDecodeError, Exception) as e:
        logger.debug(f"Response is not JSON, using raw text: {e}")

    # Extract products từ tool results
    if hasattr(result, "messages"):
        tool_products = _extract_products_from_messages(result.messages)
        # Merge và deduplicate
        seen_skus = {p.get("sku") for p in product_ids if isinstance(p, dict) and "sku" in p}
        for product in tool_products:
            if isinstance(product, dict) and product.get("sku") not in seen_skus:
                product_ids.append(product)
                seen_skus.add(product.get("sku"))

    return ai_text, product_ids


def _extract_products_from_messages(messages: list) -> list[dict]:
    """Extract products từ Agno tool messages."""
    products = []
    seen_skus = set()

    for msg in messages:
        if not (hasattr(msg, "content") and isinstance(msg.content, str)):
            continue

        try:
            tool_result = json.loads(msg.content)
            if tool_result.get("status") != "success":
                continue

            # Handle multi-search format
            if "results" in tool_result:
                for result_item in tool_result["results"]:
                    products.extend(_parse_products(result_item.get("products", []), seen_skus))
            # Handle single search format
            elif "products" in tool_result:
                products.extend(_parse_products(tool_result["products"], seen_skus))

        except (json.JSONDecodeError, KeyError, TypeError) as e:
            logger.debug(f"Skip invalid tool message: {e}")
            continue

    return products


def _parse_products(products: list[dict], seen_skus: set[str]) -> list[dict]:
    """Parse và format products, skip duplicates."""
    parsed = []
    for product in products:
        if not isinstance(product, dict):
            continue
        sku = product.get("internal_ref_code")
        if not sku or sku in seen_skus:
            continue

        seen_skus.add(sku)
        parsed.append({
            "sku": sku,
            "name": product.get("magento_product_name", ""),
            "price": product.get("price_vnd", 0),
            "sale_price": product.get("sale_price_vnd"),
            "url": product.get("magento_url_key", ""),
            "thumbnail_image_url": product.get("thumbnail_image_url", ""),
        })
    return parsed

