"""
Load Test Manager - Chạy Locust programmatically
Singleton service để quản lý load testing cho APIs
"""

import logging
import threading
import time
from dataclasses import dataclass, asdict
from enum import Enum
from typing import Any

from locust import HttpUser, between, task
from locust.env import Environment

logger = logging.getLogger(__name__)


class TestStatus(str, Enum):
    """Trạng thái của load test"""
    IDLE = "idle"
    RUNNING = "running"
    STOPPING = "stopping"
    STOPPED = "stopped"


class TestType(str, Enum):
    """Loại test"""
    CHAT_MOCK = "chat_mock"    # Mock API không tốn tiền ⭐ Khuyên dùng
    CHAT_REAL = "chat_real"    # Real API - TỐN TIỀN!
    HISTORY = "history"         # Test History API (Postgres)
    DB_SEARCH = "db_search"     # Test StarRocks DB (NO LLM cost) 🔥


@dataclass
class LoadTestConfig:
    """Config cho load test"""
    target_url: str
    num_users: int = 10
    spawn_rate: int = 2
    duration_seconds: int = 60
    test_type: str = TestType.CHAT_MOCK


@dataclass
class LoadTestMetrics:
    """Metrics realtime với percentiles"""
    status: str
    total_requests: int = 0
    current_rps: float = 0.0
    avg_response_time_ms: float = 0.0
    min_response_time_ms: float = 0.0
    max_response_time_ms: float = 0.0
    # Percentiles (quan trọng cho phân tích performance)
    p50_response_time_ms: float = 0.0  # Median - 50% requests nhanh hơn
    p90_response_time_ms: float = 0.0  # 90% requests nhanh hơn
    p95_response_time_ms: float = 0.0  # 95% requests nhanh hơn
    p99_response_time_ms: float = 0.0  # 99% requests nhanh hơn (worst case)
    failure_rate: float = 0.0
    active_users: int = 0
    elapsed_seconds: int = 0


# ==================== LOCUST USER CLASSES ====================

class ChatMockUser(HttpUser):
    """Test Mock Chat API (KHÔNG tốn tiền OpenAI)"""
    wait_time = between(0.5, 2)
    
    # Fix Windows FD limit: Disable connection pooling
    connection_timeout = 1.0
    network_timeout = 10.0

    def on_start(self):
        """Giảm pool size xuống 1 connection per user"""
        from requests.adapters import HTTPAdapter
        adapter = HTTPAdapter(pool_connections=1, pool_maxsize=1)
        self.client.mount("http://", adapter)
        self.client.mount("https://", adapter)

    @task
    def send_chat_message(self):
        # Close connection sau mỗi request
        response = self.client.post(
            "/api/test/chat-mock",
            json={
                "user_query": "Cho em xem áo sơ mi nam",
                "user_id": f"loadtest_{self.environment.runner.user_count}"
            },
            name="POST /test/chat-mock"
        )
        response.close()  # Force close ngay


class ChatRealUser(HttpUser):
    """Test Real Chat API (TỐN TIỀN - cẩn thận!)"""
    wait_time = between(1, 3)
    
    # Fix Windows FD limit
    connection_timeout = 1.0
    network_timeout = 30.0

    def on_start(self):
        from requests.adapters import HTTPAdapter
        adapter = HTTPAdapter(pool_connections=1, pool_maxsize=1)
        self.client.mount("http://", adapter)
        self.client.mount("https://", adapter)

    @task
    def send_chat_message(self):
        response = self.client.post(
            "/api/agent/chat",
            json={
                "user_query": "Cho em xem áo sơ mi nam",
                "user_id": f"loadtest_{self.environment.runner.user_count}"
            },
            name="POST /agent/chat"
        )
        response.close()


class HistoryUser(HttpUser):
    """Test History API"""
    wait_time = between(0.2, 1)
    
    # Fix Windows FD limit
    connection_timeout = 1.0
    network_timeout = 10.0

    def on_start(self):
        from requests.adapters import HTTPAdapter
        adapter = HTTPAdapter(pool_connections=1, pool_maxsize=1)
        self.client.mount("http://", adapter)
        self.client.mount("https://", adapter)

    @task
    def get_history(self):
        user_id = f"test_user_001"
        response = self.client.get(
            f"/history/{user_id}?limit=20",
            name="GET /history"
        )
        response.close()


class DBSearchUser(HttpUser):
    """Test StarRocks DB Query Performance (NO LLM cost)"""
    wait_time = between(0.3, 1.5)
    
    # Fix Windows FD limit
    connection_timeout = 1.0
    network_timeout = 15.0

    def on_start(self):
        from requests.adapters import HTTPAdapter
        adapter = HTTPAdapter(pool_connections=1, pool_maxsize=1)
        self.client.mount("http://", adapter)
        self.client.mount("https://", adapter)

    @task
    def search_products(self):
        response = self.client.post(
            "/api/test/db-search",
            json={
                "user_query": "áo sơ mi nam",
                "user_id": f"loadtest_{self.environment.runner.user_count}"
            },
            name="POST /test/db-search"
        )
        response.close()


# ==================== LOAD TEST MANAGER ====================

class LoadTestManager:
    """Singleton manager cho load testing"""

    def __init__(self):
        self.status = TestStatus.IDLE
        self.environment: Environment | None = None
        self.runner_thread: threading.Thread | None = None
        self.start_time: float | None = None
        self.config: LoadTestConfig | None = None
        self._lock = threading.Lock()

    def start_test(self, config_dict: dict) -> dict:
        """Bắt đầu load test"""
        with self._lock:
            if self.status == TestStatus.RUNNING:
                return {"error": "Test is already running"}

            try:
                self.config = LoadTestConfig(**config_dict)
                self.status = TestStatus.RUNNING
                self.start_time = time.time()

                # Chọn User class dựa trên test_type
                user_classes = {
                    TestType.CHAT_MOCK: ChatMockUser,
                    TestType.CHAT_REAL: ChatRealUser,
                    TestType.HISTORY: HistoryUser,
                    TestType.DB_SEARCH: DBSearchUser,
                }
                user_class = user_classes.get(self.config.test_type, ChatMockUser)

                # Tạo Locust Environment
                self.environment = Environment(user_classes=[user_class])
                self.environment.host = self.config.target_url

                # Bắt đầu runner trong background thread
                self.runner_thread = threading.Thread(
                    target=self._run_test,
                    daemon=True
                )
                self.runner_thread.start()

                logger.info(f"✅ Load test started: {self.config.test_type} | {self.config.num_users} users")
                return {"status": "started", "config": asdict(self.config)}

            except Exception as e:
                self.status = TestStatus.IDLE
                logger.error(f"Failed to start test: {e}")
                return {"error": str(e)}

    def _run_test(self):
        """Chạy test trong background thread"""
        try:
            runner = self.environment.create_local_runner()
            
            # Spawn users
            runner.start(
                user_count=self.config.num_users,
                spawn_rate=self.config.spawn_rate
            )

            # Chạy trong duration
            time.sleep(self.config.duration_seconds)

            # Stop gracefully
            runner.quit()
            
            with self._lock:
                self.status = TestStatus.STOPPED

            logger.info("✅ Load test completed")

        except Exception as e:
            logger.error(f"Error during test: {e}")
            with self._lock:
                self.status = TestStatus.STOPPED

    def stop_test(self) -> dict:
        """Dừng test đang chạy"""
        with self._lock:
            if self.status != TestStatus.RUNNING:
                return {"error": "No test is running"}

            self.status = TestStatus.STOPPING
            
            if self.environment and self.environment.runner:
                self.environment.runner.quit()

            self.status = TestStatus.STOPPED
            logger.info("🛑 Load test stopped by user")
            return {"status": "stopped"}

    def get_metrics(self) -> dict:
        """Lấy metrics hiện tại với percentiles"""
        with self._lock:
            if not self.environment or not self.environment.runner:
                return asdict(LoadTestMetrics(status=self.status))

            stats = self.environment.runner.stats
            total_stats = stats.total

            # Tính elapsed time
            elapsed = int(time.time() - self.start_time) if self.start_time else 0

            # Lấy percentiles từ Locust (trả về dict với key là percentile)
            try:
                # get_response_time_percentile trả về dict: {0.5: 123, 0.9: 456, ...}
                percentiles_dict = total_stats.get_response_time_percentile(0.5, 0.9, 0.95, 0.99) or {}
                p50 = percentiles_dict.get(0.5, 0)
                p90 = percentiles_dict.get(0.9, 0)
                p95 = percentiles_dict.get(0.95, 0)
                p99 = percentiles_dict.get(0.99, 0)
            except Exception as e:
                logger.warning(f"Failed to get percentiles: {e}")
                p50 = p90 = p95 = p99 = 0

            metrics = LoadTestMetrics(
                status=self.status,
                total_requests=total_stats.num_requests,
                current_rps=round(total_stats.current_rps, 2),
                avg_response_time_ms=round(total_stats.avg_response_time, 2),
                min_response_time_ms=round(total_stats.min_response_time or 0, 2),
                max_response_time_ms=round(total_stats.max_response_time or 0, 2),
                # Percentiles
                p50_response_time_ms=round(p50, 2) if p50 else 0,
                p90_response_time_ms=round(p90, 2) if p90 else 0,
                p95_response_time_ms=round(p95, 2) if p95 else 0,
                p99_response_time_ms=round(p99, 2) if p99 else 0,
                failure_rate=round(total_stats.fail_ratio, 4),
                active_users=self.environment.runner.user_count,
                elapsed_seconds=elapsed
            )

            return asdict(metrics)

    def is_running(self) -> bool:
        """Check xem test có đang chạy không"""
        return self.status == TestStatus.RUNNING


# ==================== SINGLETON ====================

_instance: LoadTestManager | None = None


def get_load_test_manager() -> LoadTestManager:
    """Get singleton instance"""
    global _instance
    if _instance is None:
        _instance = LoadTestManager()
    return _instance
