# 🌊 STREAMING BEHAVIOR EXPLAINED

## ❓ TẠI SAO VẪN CHẬM MẶC DÙ ĐÃ BẬT STREAMING?

### ✅ HIỆN TẠI - STREAMING ĐÃ BẬT:

1. **LLM Factory** ([llm_factory.py](../common/llm_factory.py#L87)):
   ```python
   llm = create_llm(model_name=..., streaming=True)  # ✅ BẬT
   ```

2. **Controller** ([controller.py](controller.py#L167)):
   ```python
   async for event in graph.astream(initial_state, config=exec_config):  # ✅ DÙNG ASTREAM
   ```

3. **Regex Early Break** ([controller.py](controller.py#L186)):
   ```python
   if product_match:
       # Bắt được product_ids → BREAK ngay!
       break  # ✅ KHÔNG ĐỢI user_insight
   ```

---

## 🔍 VẤN ĐỀ THỰC SỰ:

### **LangGraph astream() ≠ Token Streaming**

`graph.astream()` stream **EVENTS** (node completions), KHÔNG phải **TOKENS**:

```
graph.astream() tạo ra các events:
├─ Event 1: Tool node hoàn thành → {"messages": [...]}
├─ Event 2: LLM node hoàn thành → {"ai_response": AIMessage(...)}  ← TOÀN BỘ RESPONSE MỘT LẦN!
└─ Event 3: Agent node hoàn thành → {"messages": [...]}
```

**LLM response được stream BÊN TRONG node**, nhưng `graph.astream()` chỉ emit event SAU KHI node xong!

---

## ⏱️ TIMELINE THỰC TẾ:

```
t=0s:    Client gửi request
t=0-2s:  Tool execution (DB query)
t=2-12s: LLM streaming tokens (INTERNAL) ← ĐANG STREAM NHƯNG KHÔNG VISIBLE!
t=12s:   LLM node hoàn thành → graph.astream() emit event
t=12s:   Regex match product_ids → BREAK
t=12s:   Response trả về client
```

**Latency**: ~12s

---

## 💡 TẠI SAO KHÔNG BREAK SỚM HƠN?

Vì:
1. **Event chỉ emit SAU KHI LLM node xong**
2. LLM node chứa TOÀN BỘ JSON response (ai_response + product_ids + user_insight)
3. Không thể break giữa chừng vì event chưa được emit

---

## 🎯 GIẢI PHÁP ĐỂ STREAM THỰC SỰ:

### **Option 1: Custom Streaming Callback** (Phức tạp)
```python
from langchain.callbacks.base import BaseCallbackHandler

class TokenStreamCallback(BaseCallbackHandler):
    def on_llm_new_token(self, token: str, **kwargs):
        # Stream từng token ra client
        yield token
```

### **Option 2: SSE Endpoint** (Chuẩn nhất)
```python
@router.get("/api/agent/chat-stream")
async def chat_stream(request: Request):
    async def event_generator():
        async for event in graph.astream(...):
            if "ai_response" in event:
                yield f"data: {json.dumps(event)}\n\n"
    
    return StreamingResponse(event_generator(), media_type="text/event-stream")
```

### **Option 3: WebSocket** (Real-time)
```python
@router.websocket("/ws/chat")
async def websocket_chat(websocket: WebSocket):
    await websocket.accept()
    async for event in graph.astream(...):
        await websocket.send_json(event)
```

---

## ✅ CODE HIỆN TẠI LÀ TỐI ƯU NHẤT (TRONG GIỚI HẠN)

**Streaming + Early Break ĐÃ ĐÚNG:**
- ✅ Break ngay khi có product_ids
- ✅ User_insight xử lý background
- ✅ Không đợi full response

**NHƯNG:**
- ❌ Không thể break sớm hơn vì event chưa emit
- ❌ Client vẫn phải đợi LLM xong (~12s)

---

## 📊 SO SÁNH LATENCY:

| Method | Latency | Complexity |
|--------|---------|------------|
| **Current (RESTful + Internal Stream)** | ~12s | ⭐ Simple |
| **SSE Streaming** | ~8s (stream chunks) | ⭐⭐ Medium |
| **WebSocket** | ~5s (real-time) | ⭐⭐⭐ Complex |

---

## 🚀 KẾT LUẬN:

**Code ĐÃ TỐI ƯU TỐI ĐA trong RESTful context!**

Để giảm latency thêm, cần:
1. **Switch sang SSE/WebSocket** (requires client changes)
2. **Faster LLM model** (gpt-4o-mini thay vì gpt-5-nano)
3. **Cache hit** (< 100ms)

**Current implementation: 12s → Optimized: 8-10s (SSE) hoặc 5-7s (WebSocket)**
