import logging
import time

from common.embedding_service import create_embedding_async

logger = logging.getLogger(__name__)



def _get_price_clauses(params, sql_params: list) -> list[str]:
    """Lọc theo giá (Parameterized)."""
    clauses = []
    p_min = getattr(params, "price_min", None)
    if p_min is not None:
        clauses.append("sale_price >= %s")
        sql_params.append(p_min)
    p_max = getattr(params, "price_max", None)
    if p_max is not None:
        clauses.append("sale_price <= %s")
        sql_params.append(p_max)
    return clauses


def _get_metadata_clauses(params, sql_params: list) -> list[str]:
    """Xây dựng điều kiện lọc từ metadata (Parameterized)."""
    clauses = []

    # 1. Exact Match
    exact_fields = [
        ("gender_by_product", "gender_by_product"),
        ("age_by_product", "age_by_product"),
        ("form_neckline", "form_neckline"), 
    ]
    for param_name, col_name in exact_fields:
        val = getattr(params, param_name, None)
        if val:
            clauses.append(f"{col_name} = %s")
            sql_params.append(val)

    # 2. Partial Match (LIKE)
    partial_fields = [
        ("season", "season"),
        ("material_group", "material_group"),
        ("product_line_vn", "product_line_vn"),
        ("style", "style"),
        ("fitting", "fitting"),
        ("form_neckline", "form_neckline"),
        ("form_sleeve", "form_sleeve"),
    ]
    for param_name, col_name in partial_fields:
        val = getattr(params, param_name, None)
        if val:
            clauses.append(f"LOWER({col_name}) LIKE %s")
            sql_params.append(f"%{val.lower()}%")

    return clauses


def _get_special_clauses(params, sql_params: list) -> list[str]:
    """Các trường hợp đặc biệt: Mã sản phẩm, Màu sắc."""
    clauses = []
    # Mã sản phẩm / SKU
    m_code = getattr(params, "magento_ref_code", None)
    if m_code:
        clauses.append("(magento_ref_code = %s OR internal_ref_code = %s)")
        sql_params.extend([m_code, m_code])

    # Màu sắc
    color = getattr(params, "master_color", None)
    if color:
        c_wildcard = f"%{color.lower()}%"
        clauses.append("(LOWER(master_color) LIKE %s OR LOWER(product_color_name) LIKE %s)")
        sql_params.extend([c_wildcard, c_wildcard])
    return clauses


async def build_starrocks_query(params, query_vector: list[float] | None = None) -> tuple[str, list]:
    """
    Build SQL query với Parameterized Query để tránh SQL Injection.
    Returns: (sql_string, params_list)
    """

    # ============================================================
    # CASE 1: CODE SEARCH
    # ============================================================
    magento_code = getattr(params, "magento_ref_code", None)
    if magento_code:
        logger.info(f"🎯 [CODE SEARCH] Direct search by code: {magento_code}")
        

        sql = """
        SELECT 
            internal_ref_code,
            description_text_full,
            sale_price,
            original_price,
            discount_amount,
            product_line_vn,
            product_line_en,
            1.0 as max_score
        FROM shared_source.magento_product_dimension_with_text_embedding
        """
        return sql, [magento_code, magento_code]

    # ============================================================
    # CASE 2: HYDE SEARCH - Semantic Vector Search
    # ============================================================
    logger.info("🚀 [HYDE RETRIEVER] Starting semantic vector search...")
    
    query_text = getattr(params, "description", None)
    if query_text and query_vector is None:
        emb_start = time.time()
        query_vector = await create_embedding_async(query_text)
        logger.info(f"⏱️ [TIMER] Single HyDE Embedding: {(time.time() - emb_start) * 1000:.2f}ms")

    if not query_vector:
        logger.warning("⚠️ No vector found, returning empty query.")
        return "", []

    # Vector params
    v_str = "[" + ",".join(str(v) for v in query_vector) + "]"
    
    # Collect Params
    price_params: list = []
    price_clauses = _get_price_clauses(params, price_params)
    
    where_filter = ""
    if price_clauses:
        where_filter = " AND " + " AND ".join(price_clauses)
        logger.info(f"💰 [PRICE FILTER] Applied: {where_filter}")

    # Build SQL
    # NOTE: Vector v_str is safe (generated from floats) so f-string is OK here.
    # Using %s for vector list string might cause StarRocks to treat it as string literal '[...]' instead of array.
    sql = f"""
    WITH top_matches AS (
        SELECT /*+ SET_VAR(ann_params='{{"ef_search":128}}') */
            internal_ref_code,
            product_color_code,
            description_text_full,
            sale_price,
            original_price,
            discount_amount,
            approx_cosine_similarity(vector, {v_str}) as similarity_score
        FROM shared_source.magento_product_dimension_with_text_embedding
        ORDER BY similarity_score DESC
        LIMIT 200
    )
    SELECT 
        internal_ref_code,
        MAX_BY(description_text_full, similarity_score) as description_text_full,
        MAX_BY(sale_price, similarity_score) as sale_price,
        MAX_BY(original_price, similarity_score) as original_price,
        MAX_BY(discount_amount, similarity_score) as discount_amount,
        MAX(similarity_score) as max_score
    FROM top_matches
    WHERE 1=1 {where_filter}
    GROUP BY internal_ref_code
    ORDER BY max_score DESC
    LIMIT 70
    """
    
    # Return sql and params (params only contains filter values now, not the vector)
    return sql, price_params

