"""
CANIFA Data Retrieval Tool - Tối giản cho Agentic Workflow.
Hỗ trợ Hybrid Search: Semantic (Vector) + Metadata Filter.
"""

import json
import logging
from decimal import Decimal

from langchain_core.tools import tool
from pydantic import BaseModel, Field

from agent.tools.product_search_helpers import build_starrocks_query
from common.starrocks_connection import StarRocksConnection

logger = logging.getLogger(__name__)


class DecimalEncoder(json.JSONEncoder):
    """Xử lý kiểu Decimal từ Database khi convert sang JSON."""

    def default(self, obj):
        if isinstance(obj, Decimal):
            return float(obj)
        return super().default(obj)


class SearchParams(BaseModel):
    """Cấu trúc tham số tìm kiếm mà Agent phải cung cấp, map trực tiếp với Database."""

    query: str | None = Field(
        None,
        description="Câu hỏi/mục đích tự do của user (đi chơi, dự tiệc, phỏng vấn,...) - dùng cho Semantic Search",
    )
    keywords: str | None = Field(
        None, description="Từ khóa kỹ thuật cụ thể (áo polo, quần jean,...) - dùng cho LIKE search"
    )
    internal_ref_code: str | None = Field(None, description="Mã sản phẩm (ví dụ: 1TS23S012)")
    product_color_code: str | None = Field(None, description="Mã màu sản phẩm (ví dụ: 1TS23S012-SK010)")
    product_line_vn: str | None = Field(None, description="Dòng sản phẩm (Áo phông, Quần short,...)")
    gender_by_product: str | None = Field(None, description="Giới tính: male, female")
    age_by_product: str | None = Field(None, description="Độ tuổi: adult, kids, baby, others")
    master_color: str | None = Field(None, description="Màu sắc chính (Đen/ Black, Trắng/ White,...)")
    material_group: str | None = Field(None, description="Nhóm chất liệu (Knit - Dệt Kim, Woven - Dệt Thoi,...)")
    season: str | None = Field(None, description="Mùa (Spring Summer, Autumn Winter)")
    style: str | None = Field(None, description="Phong cách (Basic Update, Fashion,...)")
    fitting: str | None = Field(None, description="Form dáng (Regular, Slim, Loose,...)")
    form_neckline: str | None = Field(None, description="Kiểu cổ (Crew Neck, V-neck,...)")
    form_sleeve: str | None = Field(None, description="Kiểu tay (Short Sleeve, Long Sleeve,...)")
    price_min: float | None = Field(None, description="Giá thấp nhất")
    price_max: float | None = Field(None, description="Giá cao nhất")
    action: str = Field("search", description="Hành động: 'search' (tìm kiếm) hoặc 'visual_search' (phân tích ảnh)")


from langsmith import traceable

@tool(args_schema=SearchParams)
@traceable(run_type="tool", name="data_retrieval_tool")
async def data_retrieval_tool(
    action: str = "search",
    query: str | None = None,
    keywords: str | None = None,
    internal_ref_code: str | None = None,
    product_color_code: str | None = None,
    product_line_vn: str | None = None,
    gender_by_product: str | None = None,
    age_by_product: str | None = None,
    master_color: str | None = None,
    material_group: str | None = None,
    season: str | None = None,
    style: str | None = None,
    fitting: str | None = None,
    form_neckline: str | None = None,
    form_sleeve: str | None = None,
    price_min: float | None = None,
    price_max: float | None = None,
) -> str:
    """
    Tìm kiếm sản phẩm trong database của CANIFA sử dụng tìm kiếm ngữ nghĩa (Semantic), từ khóa (Keywords) hoặc các bộ lọc thuộc tính.
    
    Cơ chế hoạt động (Hybrid Search):
    - Nếu có 'query': Hệ thống sẽ tạo vector embedding và tìm kiếm theo độ tương đồng ngữ nghĩa.
    - Nếu có 'keywords' hoặc các thuộc tính khác: Hệ thống sẽ tạo các câu lệnh SQL WHERE để lọc chính xác kết quả.
    - Kết hợp cả hai để mang lại kết quả tối ưu nhất.

    Ví dụ sử dụng (Examples):
    1. Tìm kiếm theo ý định chung:
       User: "Tìm cho mình một bộ đồ đi biển mát mẻ"
       Tool call: data_retrieval_tool(query="bộ đồ đi biển mát mẻ", gender_by_product="Female")

    2. Tìm chính xác theo loại sản phẩm và giá:
       User: "Áo polo nam dưới 400k"
       Tool call: data_retrieval_tool(keywords="áo polo", gender_by_product="Male", price_max=400000)

    3. Tìm theo mã sản phẩm cụ thể:
       User: "Check sản phẩm 8TS24W001"
       Tool call: data_retrieval_tool(internal_ref_code="8TS24W001")

    4. Kết hợp tìm kiếm sâu:
       User: "Áo khoác len mùa đông cho bé trai từ 200k đến 500k"
       Tool call: data_retrieval_tool(query="áo khoác len ấm áp", material_group="Len", age_by_product="Kids", price_min=200000, price_max=500000)
    """
    try:
        # 1. Log & Prepare Params
        # 1. Log & Prepare Params
        params = SearchParams(
            action=action,
            query=query,
            keywords=keywords,
            internal_ref_code=internal_ref_code,
            product_color_code=product_color_code,
            product_line_vn=product_line_vn,
            gender_by_product=gender_by_product,
            age_by_product=age_by_product,
            master_color=master_color,
            material_group=material_group,
            season=season,
            style=style,
            fitting=fitting,
            form_neckline=form_neckline,
            form_sleeve=form_sleeve,
            price_min=price_min,
            price_max=price_max,
        )
        _log_agent_call(params)

        # 2. Prepare Vector (Async) if needed
        query_vector = None
        if query:
            from common.embedding_service import create_embedding_async
            query_vector = await create_embedding_async(query)

        # 3. Execute Search (Async)
        sql = build_starrocks_query(params, query_vector=query_vector)
        db = StarRocksConnection()
        products = await db.execute_query_async(sql)

        if not products:
            return _handle_no_results(query, keywords)

        # 4. Format Results
        clean_products = _format_product_results(products)
        return json.dumps(
            {"status": "success", "count": len(clean_products), "products": clean_products},
            ensure_ascii=False,
            cls=DecimalEncoder,
        )

    except Exception as e:
        logger.error(f"Error in data_retrieval_tool: {e}")
        return json.dumps({"status": "error", "message": str(e)})


def _log_agent_call(params: SearchParams):
    """Log parameters for debugging."""
    filtered_params = {k: v for k, v in params.dict().items() if v is not None}
    logger.info(f"📋 Agent Tool Call - data_retrieval_tool: {json.dumps(filtered_params, ensure_ascii=False)}")


def _handle_no_results(query: str | None, keywords: str | None) -> str:
    """Return standardized no-results message."""
    logger.warning(f"No products found for search: query={query}, keywords={keywords}")
    return json.dumps(
        {
            "status": "no_results",
            "message": "Không tìm thấy sản phẩm nào phù hợp với yêu cầu. Vui lòng thử lại với từ khóa hoặc bộ lọc khác.",
        },
        ensure_ascii=False,
    )


def _format_product_results(products: list[dict]) -> list[dict]:
    """Filter and format product fields for the agent."""
    allowed_fields = {
        "internal_ref_code",
        "magento_ref_code",
        "product_color_code",
        "product_name",
        "color_code",
        "master_color",
        "product_color_name",
        "season_sale",
        "season",
        "style",
        "fitting",
        "size_scale",
        "graphic",
        "pattern",
        "weaving",
        "shape_detail",
        "form_neckline",
        "form_sleeve",
        "form_length",
        "form_waistline",
        "form_shoulderline",
        "material",
        "product_group",
        "product_line_vn",
        "unit_of_measure",
        "sale_price",
        "original_price",
        "material_group",
        "product_line_en",
        "age_by_product",
        "gender_by_product",
        "product_image_url",
        "description_text",
        "product_image_url_thumbnail",
        "product_web_url",
        "product_web_material",
    }
    return [{k: v for k, v in p.items() if k in allowed_fields} for p in products[:5]]
