"""
Fashion Q&A Agent Graph
LangGraph workflow với clean architecture.
Tất cả resources (LLM, Tools) khởi tạo trong __init__.
Sử dụng ConversationManager (Postgres) để lưu history thay vì checkpoint.
"""

import logging
from typing import Any

from langchain_core.language_models import BaseChatModel
from langchain_core.messages import HumanMessage
from langchain_core.prompts import ChatPromptTemplate, MessagesPlaceholder
from langchain_core.runnables import RunnableConfig
from langgraph.cache.memory import InMemoryCache
from langgraph.graph import END, StateGraph
from langgraph.prebuilt import ToolNode
from langgraph.types import CachePolicy

from common.llm_factory import create_llm

from .models import AgentConfig, AgentState, get_config
from .prompt import get_system_prompt
from .tools.get_tools import get_all_tools, get_collection_tools

logger = logging.getLogger(__name__)


class CANIFAGraph:
    """
    Fashion Q&A Agent Graph Manager.
    Khởi tạo tất cả resources trong __init__, các node dùng self.xxx.
    """

    def __init__(
        self,
        config: AgentConfig | None = None,
        llm: BaseChatModel | None = None,
        tools: list | None = None,
    ):
        self.config = config or get_config()
        self._compiled_graph: Any | None = None

        # Dependency Injection: Ưu tiên dùng llm/tools được truyền vào
        self.llm: BaseChatModel = llm or create_llm(
            model_name=self.config.model_name, api_key=self.config.openai_api_key, streaming=True
        )

        # Phân loại tools
        self.all_tools = tools or get_all_tools()
        self.collection_tools = get_collection_tools()  # Vẫn lấy list name để routing
        
        # Retrieval tools are logically all tools minus collection tools (conceptually, or specific list)
        # For simplicity and robust tool usage, we can bind all tools to retrieval node if needed, 
        # or separate them. The user code snippet uses `self.retrieval_tools` but passed `all_tools`.
        # Reviewing user snippet: `workflow.add_node("retrieve_tools", ToolNode(self.retrieval_tools)...`
        # But `retrieval_tools` wasn't defined in __init__ in the user snippet, likely implied.
        # I'll define retrieval_tools as all tools for now or filter if strictly needed.
        # Assuming all_tools are retrieval compatible except collection ones? 
        # Let's use all_tools for the ToolNode to be safe unless distinct behavior is needed.
        self.retrieval_tools = self.all_tools

        self.llm_with_tools = self.llm.bind_tools(self.all_tools)
        self.system_prompt = get_system_prompt()
        self.cache = InMemoryCache()

    async def _agent_node(self, state: AgentState, config: RunnableConfig) -> dict:
        """Agent node - LLM reasoning với tools và history sạch."""
        messages = state.get("messages", [])
        history = state.get("history", [])

        prompt = ChatPromptTemplate.from_messages(
            [
                ("system", self.system_prompt),
                MessagesPlaceholder(variable_name="history"),  # Long-term clean history
                MessagesPlaceholder(variable_name="messages"),  # Current turn technical messages
            ]
        )

        # 2. Xử lý Image hint (Lấy từ Config của lượt chạy này)
        transient_images = config.get("configurable", {}).get("transient_images", [])

        if transient_images and messages:
            # Removed image processing logic as requested
            pass

        # Invoke LLM
        chain = prompt | self.llm_with_tools
        response = await chain.ainvoke({"messages": messages, "history": history})

        return {"messages": [response]}

    def _should_continue(self, state: AgentState) -> str:
        """Routing: tool nodes hoặc end."""
        last_message = state["messages"][-1]

        if not hasattr(last_message, "tool_calls") or not last_message.tool_calls:
            logger.info("🏁 Agent finished")
            return "end"

        tool_names = [tc["name"] for tc in last_message.tool_calls]
        collection_names = [t.name for t in self.collection_tools]

        if any(name in collection_names for name in tool_names):
            logger.info(f"🔄 → collect_tools: {tool_names}")
            return "collect_tools"

        logger.info(f"🔄 → retrieve_tools: {tool_names}")
        return "retrieve_tools"

    def build(self) -> Any:
        """Build và compile LangGraph workflow (Không dùng Checkpointer)."""
        if self._compiled_graph is not None:
            return self._compiled_graph

        logger.info("🔨 Building LangGraph workflow (No Checkpointer)...")

        workflow = StateGraph(AgentState)

        # Nodes
        workflow.add_node("agent", self._agent_node)
        workflow.add_node("retrieve_tools", ToolNode(self.retrieval_tools), cache_policy=CachePolicy(ttl=3600))
        workflow.add_node("collect_tools", ToolNode(self.collection_tools))

        # Edges
        workflow.set_entry_point("agent")
        workflow.add_conditional_edges(
            "agent",
            self._should_continue,
            {"retrieve_tools": "retrieve_tools", "collect_tools": "collect_tools", "end": END},
        )
        workflow.add_edge("retrieve_tools", "agent")
        workflow.add_edge("collect_tools", "agent")

        # Compile WITHOUT checkpointer
        self._compiled_graph = workflow.compile(cache=self.cache)

        # ❌ KHÔNG ATTACH Langfuse callback vào compiled graph
        # ✅ Sẽ pass callback vào runtime config của mỗi lượt chạy
        logger.info("✅ Graph compiled (Langfuse callback will be per-run)")

        return self._compiled_graph

    @property
    def graph(self) -> Any:
        return self.build()


# --- Singleton & Public API ---
_instance: list[CANIFAGraph | None] = [None]  


def build_graph(config: AgentConfig | None = None, llm: BaseChatModel | None = None, tools: list | None = None) -> Any:
    """Get compiled graph (singleton)."""
    if _instance[0] is None:
        _instance[0] = CANIFAGraph(config, llm, tools)
    return _instance[0].build()


def get_graph_manager(
    config: AgentConfig | None = None, llm: BaseChatModel | None = None, tools: list | None = None
) -> CANIFAGraph:
    """Get CANIFAGraph instance."""
    if _instance[0] is None:
        _instance[0] = CANIFAGraph(config, llm, tools)
    return _instance[0]


def reset_graph() -> None:
    """Reset singleton for testing."""
    _instance[0] = None
