import json
import numpy as np
import os
from loguru import logger
import ast

# Import our FaissManager
from db.faiss_mgr import FaissManager
from config import INDEX_PATH, MAPPING_PATH

def ingest_from_json(json_path):
    if not os.path.exists(json_path):
        logger.error(f"JSON file not found at {json_path}")
        return

    logger.info(f"Reading data from {json_path}...")
    with open(json_path, 'r', encoding='utf-8') as f:
        data = json.load(f)

    # Process items
    all_items = []
    if isinstance(data, dict):
        for key in data:
            if isinstance(data[key], list):
                all_items.extend(data[key])
    elif isinstance(data, list):
        all_items = data

    logger.info(f"Found {len(all_items)} products to index.")

    faiss_mgr = FaissManager(dimension=768, index_path=INDEX_PATH, mapping_path=MAPPING_PATH)
    
    vectors = []
    skus = []

    for item in all_items:
        sku = item.get('internal_ref_code')
        vector_str = item.get('vector')
        
        if not sku or not vector_str:
            continue
            
        try:
            # Vector is stored as a string "[0.1, 0.2, ...]"
            vector = np.array(ast.literal_eval(vector_str)).astype('float32')
            
            # Normalize for similarity
            norm = np.linalg.norm(vector)
            if norm > 0:
                vector = vector / norm
            
            vectors.append(vector)
            skus.append(sku)
        except Exception as e:
            logger.error(f"Error parsing vector for SKU {sku}: {e}")

    if vectors:
        vectors_np = np.stack(vectors)
        faiss_mgr.add_vectors(vectors_np, skus)
        faiss_mgr.save()
        logger.info(f"Ingestion complete. Total items in FAISS: {faiss_mgr.index.ntotal}")
    else:
        logger.warning("No valid vectors found to index.")

if __name__ == "__main__":
    target_json = r"d:\cnf\chatbot_canifa_image\data\magento_product_dimension_with_image_embedding_202512241158.json"
    ingest_from_json(target_json)
