"""
RECOMMEND TEXT API
═══════════════════

API endpoint cho Text-based Product Recommendation
Sử dụng Hybrid Search: Semantic (Vector) Search → Keyword Filtering
"""

import logging
from fastapi import APIRouter
from pydantic import BaseModel, Field
from typing import Optional

from search.hybrid_search import hybrid_search

logger = logging.getLogger(__name__)

router = APIRouter(prefix="/api/recommend", tags=["Recommendation - Text"])


class SearchParams(BaseModel):
    """Request body cho search API"""
    query: str = Field(..., description="Search query (VD: 'áo mùa đông cho bé gái')")
    limit: int = Field(default=20, ge=1, le=100, description="Số kết quả trả về (1-100)")
    price_min: Optional[float] = Field(default=None, ge=0, description="Giá tối thiểu")
    price_max: Optional[float] = Field(default=None, ge=0, description="Giá tối đa")


@router.post("/text")
async def recommend_by_text(params: SearchParams):
    """
    🔍 HYBRID SEARCH: Semantic Search → Keyword Filtering
    
    Flow:
    1. Thêm prefix "description_text: " vào query
    2. Embed prefixed query → Vector
    3. Vector search (HNSW) → TOP 200 candidates
    4. Parse keywords → Filter (season, gender, age, product_type, color...)
    5. Return TOP N results
    
    Example queries:
    - "áo mùa đông cho bé gái"
    - "quần jean nam"
    - "váy đỏ người lớn"
    - "áo khoác phao trẻ em"
    """
    logger.info(f"Hybrid search: query='{params.query}', limit={params.limit}")
    
    result = await hybrid_search(
        query=params.query,
        limit=params.limit,
        price_min=params.price_min,
        price_max=params.price_max,
    )
    
    return result
